//
// Copyright (c) Citrix Systems, Inc.
//
/// @file RootHubPdo.cpp Emulates a Root Hub Device.
/// This is a "raw pdo" device that emulates a root hub in the standard 
/// Microsoft USB architecture. The device acts as a bus driver, enumerating a
/// child device representing the actual USB device.
//
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
#include "driver.h"
#include "UsbConfig.h"
#include <wdmguid.h>
#include <devguid.h>
#include "RootHubPdo.h"
#include <Usbioctl.h >
#include "xenusb.h"

//
// Events from the IoQueue object
//
EVT_WDF_IO_QUEUE_IO_DEVICE_CONTROL HubEvtIoDeviceControl;
EVT_WDF_IO_QUEUE_IO_INTERNAL_DEVICE_CONTROL HubEvtIoInternalDeviceControl;
EVT_WDF_IO_QUEUE_IO_STOP HubEvtIoStop;
EVT_WDF_IO_QUEUE_IO_STOP HubStatusChangeEvtIoStop;
EVT_WDF_DEVICE_FILE_CREATE  HubEvtDeviceFileCreate;
EVT_WDF_FILE_CLOSE  HubEvtFileClose;
//
// this are for debugging usbhub support
//
EVT_WDFDEVICE_WDM_IRP_PREPROCESS  RootHubPreProcessPnpIrp;
EVT_WDFDEVICE_WDM_IRP_PREPROCESS  RootHubPreProcessCreateIrp;
//
// pnp events of interest.
//
EVT_WDF_DEVICE_QUERY_REMOVE  HubEvtDeviceQueryRemove;

NTSTATUS
HubCreateChildPdo(
    IN WDFDEVICE Parent,
    IN PUSB_FDO_CONTEXT fdoContext,
    PWDFDEVICE_INIT pDeviceInit);

/**
 * @brief Create the child PDO for the root hub (bus driver) for this HBA. 
 *
 * The root hub exists just to emulate the MSFT
 * architecture and support the IOCTLs and interfaces specific
 * to Hubs while routine URBs from its child device to the 
 * parent HBA.
 *
 * The Xen bus enumerates a  "xen\\vusb" PDO, this driver instantiates a virtual USB
 * controller FDO for that PDO. The VUSB FDO enumerates a root hub "raw" PDO device, also
 * managed by this driver. The root hub "raw" PDO device enumerates a USB PDO  device.
 * 
 * @param[in] fdoContext .the context (device extenstion) for the parent controller FDO.
 * 
 * @returns NTSTATUS value for success or failure.
 * 
 */
NTSTATUS
CreateRootHubPdo(
    IN PUSB_FDO_CONTEXT fdoContext)
{
    NTSTATUS  status;
    PWDFDEVICE_INIT  pDeviceInit = NULL;

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_DEVICE,
        __FUNCTION__": %s device %p\n",
        fdoContext->FrontEndPath,
        fdoContext->WdfDevice);


    pDeviceInit = WdfPdoInitAllocate(fdoContext->WdfDevice);
    if (pDeviceInit == NULL)
    {
        TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
            __FUNCTION__": %s WdfPdoInitAllocate returned NULL\n",
            fdoContext->FrontEndPath);

        status = STATUS_INSUFFICIENT_RESOURCES;
    }
    else
    {
        status = CreateRootHubPdoWithDeviceInit(fdoContext,
            NULL,
            pDeviceInit);
    }
    return status;
}

NTSTATUS
CreateRootHubPdoWithDeviceInit(
    IN PUSB_FDO_CONTEXT fdoContext,
    _In_opt_ PWDF_CHILD_IDENTIFICATION_DESCRIPTION_HEADER IdentificationDescription,
    IN PWDFDEVICE_INIT pDeviceInit)
{
    NTSTATUS  status;
    WDFDEVICE  hChild = NULL;
    WDF_OBJECT_ATTRIBUTES  pdoAttributes;

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_DEVICE,
        __FUNCTION__": %s device %p Id %p\n",
        fdoContext->FrontEndPath,
        fdoContext->WdfDevice,
        IdentificationDescription);

    do
    {        
        //
        // Are any of the next 4 steps required?
        //
        WdfDeviceInitSetDeviceType(pDeviceInit, FILE_DEVICE_USB);
        WdfDeviceInitSetIoType(pDeviceInit, WdfDeviceIoDirect);
        WdfDeviceInitSetCharacteristics(pDeviceInit,
            FILE_DEVICE_SECURE_OPEN|FILE_AUTOGENERATED_DEVICE_NAME,
            TRUE);
        WdfDeviceInitSetDeviceClass(pDeviceInit, &GUID_DEVCLASS_USB);
        //
        // NB the id strings have to be in the form "BUS\\ID" or 
        // your PDO ends up stuck in init state.
        //
        DECLARE_CONST_UNICODE_STRING(roothubidMSFT, L"USB\\ROOT_HUB\0");
        //
        // Actual ROOT_HUB strings:
        // DECLARE_CONST_UNICODE_STRING(mostHwId, L"USB\\ROOT_HUB&VID8086&PID7020&REV0001");
        // DECLARE_CONST_UNICODE_STRING(middleHwId, L"USB\\ROOT_HUB&VID8086&PID7020");
        // DECLARE_CONST_UNICODE_STRING(leastHwId, L"USB\\ROOT_HUB");
        status = WdfPdoInitAssignDeviceID(pDeviceInit, &roothubidMSFT);
        if (!NT_SUCCESS(status))
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfPdoInitAssignDeviceID for HUB PDO error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }
        status = WdfPdoInitAddHardwareID(pDeviceInit, &roothubidMSFT);
        if (!NT_SUCCESS(status))
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfPdoInitAddHardwareID for HUB PDO error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }
        //
        // do we need this?
        //
        status = WdfPdoInitAddCompatibleID(pDeviceInit, &roothubidMSFT);
        if (!NT_SUCCESS(status))
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfPdoInitAddCompatibleID for HUB PDO error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }

        DECLARE_CONST_UNICODE_STRING(rootHubLocation, L"0");
        DECLARE_CONST_UNICODE_STRING(roothubtext, L"Xen Virtual Root Hub\0");
        status = WdfPdoInitAddDeviceText(pDeviceInit,
                                 &roothubtext,
                                 &rootHubLocation,
                                 0x409);
        if (!NT_SUCCESS(status))
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfPdoInitAddDeviceText for HUB PDO error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }
        WdfPdoInitSetDefaultLocale(pDeviceInit, 0x409);
        //
        // Add instance and container ids (usbhub?)
        //
        DECLARE_UNICODE_STRING_SIZE(instanceID, 40);
        status = RtlUnicodeStringPrintf(&instanceID,
            L"%02.2d%02.d%02.2d%d",
            L'X', L'e',L'n',
            fdoContext->Port);
        if (!NT_SUCCESS(status))
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s RtlUnicodeStringPrintf for instance id error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }

        status = WdfPdoInitAssignInstanceID(pDeviceInit,
            &instanceID );
        if (!NT_SUCCESS(status))
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfPdoInitAssignInstanceID error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }
        //
        // establish a request context. 
        // Note: because urb's are *forwarded* to the parent device for processing
        // the child device has to establish the context space used for the request.
        // It is a bit of overkill to require all requests to have a context space
        // rather than only urb requests, but the alternative is to deal with resource
        // allocation failure here rather than let the framwework deal with it.
        // Note also: this PDO device doesn't use the context.
        //
        WDF_OBJECT_ATTRIBUTES   requestAttributes;
        WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE(&requestAttributes, FDO_REQUEST_CONTEXT);
        WdfDeviceInitSetRequestAttributes(pDeviceInit, &requestAttributes);

        //
        // add a preprocess callback for IRP_MJ_PNP.
        // xxx adjust as need - initially everything
        // UCHAR MinorFunctionTable[1] = {IRP_MN_QUERY_INTERFACE};
        //
        status = WdfDeviceInitAssignWdmIrpPreprocessCallback(
            pDeviceInit,
            RootHubPreProcessPnpIrp,
            IRP_MJ_PNP,
            NULL, // MinorFunctionTable,
            0);
        if (!NT_SUCCESS(status))
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfDeviceInitAssignWdmIrpPreprocessCallback IRP_MJ_PNP error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }
        status = WdfDeviceInitAssignWdmIrpPreprocessCallback(
            pDeviceInit,
            RootHubPreProcessCreateIrp,
            IRP_MJ_CREATE,
            NULL, // MinorFunctionTable,
            0);
        if (!NT_SUCCESS(status))
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfDeviceInitAssignWdmIrpPreprocessCallback IRP_MJ_CREATE error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }
        // Allow PDO to requeue requests to its parent device.
        //
        WdfPdoInitAllowForwardingRequestToParent(pDeviceInit);

        WDF_PNPPOWER_EVENT_CALLBACKS    pnpPowerCallbacks;
        WDF_PNPPOWER_EVENT_CALLBACKS_INIT(&pnpPowerCallbacks);
        pnpPowerCallbacks.EvtDeviceD0Entry = HubEvtDeviceD0Entry;
        pnpPowerCallbacks.EvtDeviceD0Exit  = HubEvtDeviceD0Exit;
        pnpPowerCallbacks.EvtDeviceSurpriseRemoval = HubEvtDeviceSurpriseRemoval;
        pnpPowerCallbacks.EvtDeviceQueryRemove = HubEvtDeviceQueryRemove;

        WdfDeviceInitSetPnpPowerEventCallbacks(pDeviceInit, &pnpPowerCallbacks);

        WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE(&pdoAttributes, USB_HUB_PDO_CONTEXT);

        //
        // Add create/close handlers
        //
        WDF_OBJECT_ATTRIBUTES   fileAttributes;
        WDF_OBJECT_ATTRIBUTES_INIT(&fileAttributes);
        fileAttributes.SynchronizationScope = WdfSynchronizationScopeNone;
        WDF_FILEOBJECT_CONFIG FileObjectConfig;
        WDF_FILEOBJECT_CONFIG_INIT(
            &FileObjectConfig,
            HubEvtDeviceFileCreate,
            HubEvtFileClose,
            WDF_NO_EVENT_CALLBACK);

        WdfDeviceInitSetFileObjectConfig(
            pDeviceInit,
            &FileObjectConfig,
            &fileAttributes);

        status = WdfDeviceCreate(
            &pDeviceInit,
            &pdoAttributes,
            &hChild);

        if (!NT_SUCCESS(status)) 
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfDeviceCreate for HUB PDO error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }
        pDeviceInit = NULL; // don't clean this up on error.
        
        PUSB_HUB_PDO_CONTEXT hubContext = DeviceGetHubPdoContext(hChild);
        RtlZeroMemory(hubContext, sizeof(*hubContext));
        hubContext->WdfDevice = hChild;
        hubContext->Parent = fdoContext->WdfDevice;
        hubContext->ParentQueue = fdoContext->UrbQueue;
        hubContext->Port = 1; // only one port, always port 1.
        hubContext->HubConfig.DeviceDescriptor.bLength = sizeof(USB_DEVICE_DESCRIPTOR); // 18
        hubContext->HubConfig.DeviceDescriptor.bDescriptorType = USB_DEVICE_DESCRIPTOR_TYPE;
        hubContext->HubConfig.DeviceDescriptor.bcdUSB = 0x0200;
        hubContext->HubConfig.DeviceDescriptor.bDeviceClass = USB_DEVICE_CLASS_HUB;
        hubContext->HubConfig.DeviceDescriptor.bDeviceSubClass = 0;
        hubContext->HubConfig.DeviceDescriptor.bDeviceProtocol = 1;
        hubContext->HubConfig.DeviceDescriptor.bMaxPacketSize0 = 64; 
        hubContext->HubConfig.DeviceDescriptor.idVendor = 0;
        hubContext->HubConfig.DeviceDescriptor.idProduct = 0;
        hubContext->HubConfig.DeviceDescriptor.bcdDevice = 0x0101;
        hubContext->HubConfig.DeviceDescriptor.iManufacturer = 3;
        hubContext->HubConfig.DeviceDescriptor.iProduct = 2;
        hubContext->HubConfig.DeviceDescriptor.iSerialNumber = 1;
        hubContext->HubConfig.DeviceDescriptor.bNumConfigurations = 1;

        hubContext->HubConfig.ConfigurationDescriptor.bLength = sizeof(USB_CONFIGURATION_DESCRIPTOR); // 9
        hubContext->HubConfig.ConfigurationDescriptor.bDescriptorType = USB_CONFIGURATION_DESCRIPTOR_TYPE;
        hubContext->HubConfig.ConfigurationDescriptor.wTotalLength = sizeof(USB_CONFIGURATION_DESCRIPTOR) 
            + sizeof(USB_INTERFACE_DESCRIPTOR) + sizeof(USB_ENDPOINT_DESCRIPTOR); // ?? 25
        hubContext->HubConfig.ConfigurationDescriptor.bNumInterfaces = 1;
        hubContext->HubConfig.ConfigurationDescriptor.bConfigurationValue = 1;
        hubContext->HubConfig.ConfigurationDescriptor.iConfiguration = 0;
        hubContext->HubConfig.ConfigurationDescriptor.bmAttributes = USB_CONFIG_SELF_POWERED;
        hubContext->HubConfig.ConfigurationDescriptor.MaxPower = 0;

        hubContext->HubConfig.InterfaceDescriptor.bLength = sizeof(USB_INTERFACE_DESCRIPTOR); // 9
        hubContext->HubConfig.InterfaceDescriptor.bDescriptorType = USB_INTERFACE_DESCRIPTOR_TYPE;
        hubContext->HubConfig.InterfaceDescriptor.bInterfaceNumber = 0;
        hubContext->HubConfig.InterfaceDescriptor.bAlternateSetting = 0;
        hubContext->HubConfig.InterfaceDescriptor.bNumEndpoints = 1;
        hubContext->HubConfig.InterfaceDescriptor.bInterfaceClass = USB_DEVICE_CLASS_HUB;
        hubContext->HubConfig.InterfaceDescriptor.bInterfaceSubClass = 0;
        hubContext->HubConfig.InterfaceDescriptor.bInterfaceProtocol = 0;
        hubContext->HubConfig.InterfaceDescriptor.iInterface = 0;

        hubContext->HubConfig.EndpointDescriptor.bLength = sizeof(USB_ENDPOINT_DESCRIPTOR); // 7
        hubContext->HubConfig.EndpointDescriptor.bDescriptorType = USB_ENDPOINT_DESCRIPTOR_TYPE;
        hubContext->HubConfig.EndpointDescriptor.bEndpointAddress = USB_ENDPOINT_DIRECTION_IN(1);
        hubContext->HubConfig.EndpointDescriptor.bmAttributes = USB_ENDPOINT_TYPE_INTERRUPT;
        hubContext->HubConfig.EndpointDescriptor.wMaxPacketSize = 2;
        hubContext->HubConfig.EndpointDescriptor.bInterval = 0xff;

        hubContext->HubConfig.PipeDescriptor.endpoint = &hubContext->HubConfig.EndpointDescriptor;
        hubContext->HubConfig.PipeDescriptor.interfaceDescriptor = &hubContext->HubConfig.InterfaceDescriptor;
        hubContext->HubConfig.PipeDescriptor.valid = TRUE;

        hubContext->HubConfig.HubDescriptor.bDescriptorLength = 9;
        hubContext->HubConfig.HubDescriptor.bDescriptorType = USB_20_HUB_DESCRIPTOR_TYPE;
        hubContext->HubConfig.HubDescriptor.bNumberOfPorts = 1;
        hubContext->HubConfig.HubDescriptor.wHubCharacteristics = 0x1b;
        hubContext->HubConfig.HubDescriptor.bPowerOnToPowerGood = 0;
        hubContext->HubConfig.HubDescriptor.bHubControlCurrent = 0;
        hubContext->HubConfig.HubDescriptor.bRemoveAndPowerMask[0] = 0;
        hubContext->HubConfig.HubDescriptor.bRemoveAndPowerMask[1] = 0xff;

        hubContext->HubConfig.ConfigInfo.m_configurationDescriptor = 
            &hubContext->HubConfig.ConfigurationDescriptor;
        hubContext->HubConfig.InterfaceArray[0] = &hubContext->HubConfig.InterfaceDescriptor;
        hubContext->HubConfig.ConfigInfo.m_interfaceDescriptors = 
            hubContext->HubConfig.InterfaceArray;
        hubContext->HubConfig.ConfigInfo.m_pipeDescriptors = 
            &hubContext->HubConfig.PipeDescriptor;
        hubContext->HubConfig.ConfigInfo.m_numInterfaces = 1;
        hubContext->HubConfig.ConfigInfo.m_numEndpoints = 1;
        //
        // add device interface support (forward to parent)
        // for USBDI controller interface.
        // Prefast complains if the PINTERFACE value is NULL,
        // However the docs are sort of clear:
        // "Additionally, the Interface member can be NULL if the value of the
        // SendQueryToParentStack member is TRUE."
        //
        WDF_QUERY_INTERFACE_CONFIG interfaceConfig;
#pragma warning(suppress: 6387)
        WDF_QUERY_INTERFACE_CONFIG_INIT(&interfaceConfig,
            NULL,
            &USB_BUS_INTERFACE_USBDI_GUID,
            NULL);
        interfaceConfig.SendQueryToParentStack = TRUE;
        status = WdfDeviceAddQueryInterface(hChild,
            &interfaceConfig);
        if (!NT_SUCCESS(status))
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfDeviceAddQueryInterface for HUB PDO error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }
        //
        // Set up the io processing for this device.
        //
        status = HubQueueInitialize(hChild);
        if (!NT_SUCCESS(status)) 
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s HubQueueInitialize for HUB PDO error %x\n",
                fdoContext->FrontEndPath,
                status);
            break;
        }
        //
        // @todo possibly wrong if we are connecting to usbhub.
        //
        status = WdfDeviceCreateDeviceInterface(
            hChild,
            &GUID_DEVINTERFACE_USB_HUB,
            NULL);

        if (!NT_SUCCESS(status)) 
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfDeviceCreateDeviceInterface for device %p error %x\n",
                fdoContext->FrontEndPath,
                hChild,
                status);
            break;
        }

        WDF_OBJECT_ATTRIBUTES_INIT(&fileAttributes);    
        fileAttributes.ParentObject = fdoContext->WdfDevice;
        status = WdfStringCreate(NULL, &fileAttributes, &fdoContext->hubsymlink);
        if (!NT_SUCCESS(status)) 
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfStringCreate for device %p error %x\n",
                fdoContext->FrontEndPath,
                hChild,
                status);
            break;
        }

        WDF_DEVICE_PNP_CAPABILITIES   pnpCaps;
        WDF_DEVICE_PNP_CAPABILITIES_INIT(&pnpCaps);

        pnpCaps.SurpriseRemovalOK = WdfTrue;
        pnpCaps.Removable = WdfFalse;
        // If UINumber is unknown, or if supplying a number would not assist 
        // the user in identifying the device's location, the driver sets 
        // this value to -1.
        pnpCaps.UINumber = (ULONG) -1; 
        WdfDeviceSetPnpCapabilities(hChild, &pnpCaps);

        PNP_BUS_INFORMATION  busInformation;
        busInformation.BusNumber = 0;
        busInformation.BusTypeGuid = GUID_BUS_TYPE_USB;
        busInformation.LegacyBusType = PNPBus;

        WdfDeviceSetBusInformationForChildren(
            hChild,
            &busInformation);

        status = WdfFdoAddStaticChild(
            fdoContext->WdfDevice,
            hChild );

        if (!NT_SUCCESS(status)) 
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s WdfFdoAddStaticChild for Hub error %x\n",
                fdoContext->FrontEndPath,
                status);
        }

    } while (0);

    if (!NT_SUCCESS(status))
    {
        TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
                __FUNCTION__": %s device %p returning error %x pDeviceInit %p hChild %p\n",
                fdoContext->FrontEndPath,
                fdoContext->WdfDevice,
                pDeviceInit,
                hChild);
        //
        // Only free the pDeviceInit if this driver created it,
        // flagged by IdentificationDescription not being null.
        //
        if (pDeviceInit && IdentificationDescription)
        {
            WdfDeviceInitFree(pDeviceInit);
        }
        if (hChild)
        {
            WdfObjectDelete(hChild);
        }
    }
    else
    {
        // for the log:
        TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_DEVICE,
            __FUNCTION__": %s WDFDEVICE %p child PDO WDFDEVICE %p\n",
            fdoContext->FrontEndPath,
            fdoContext->WdfDevice,
            hChild);
    }

    return status;
}




NTSTATUS
RootHubPreProcessPnpIrp(
    IN WDFDEVICE Device,
    IN PIRP Irp)
{
    // log and forget
    NTSTATUS Status = Irp->IoStatus.Status;
    PIO_STACK_LOCATION IoStack = IoGetCurrentIrpStackLocation(Irp);
    PCHAR requeststring = "Unknown";
    ULONG traceLevel = TRACE_LEVEL_VERBOSE;

    switch (IoStack->MinorFunction)
    {
    case IRP_MN_START_DEVICE:
        requeststring = "IRP_MN_START_DEVICE";
        break;
    case IRP_MN_QUERY_REMOVE_DEVICE:
        traceLevel = TRACE_LEVEL_WARNING;
        requeststring = "IRP_MN_QUERY_REMOVE_DEVICE";
        break;
    case IRP_MN_REMOVE_DEVICE:
        traceLevel = TRACE_LEVEL_WARNING;
        requeststring = "IRP_MN_REMOVE_DEVICE";
        break;
    case IRP_MN_CANCEL_REMOVE_DEVICE:
        traceLevel = TRACE_LEVEL_WARNING;
        requeststring = "IRP_MN_CANCEL_REMOVE_DEVICE";
        break;
    case IRP_MN_STOP_DEVICE:
        traceLevel = TRACE_LEVEL_WARNING;
        requeststring = "IRP_MN_STOP_DEVICE";
        break;
    case IRP_MN_QUERY_STOP_DEVICE:
        traceLevel = TRACE_LEVEL_WARNING;
        requeststring = "IRP_MN_QUERY_STOP_DEVICE";
        break;
    case IRP_MN_CANCEL_STOP_DEVICE:
        traceLevel = TRACE_LEVEL_WARNING;
        requeststring = "IRP_MN_CANCEL_STOP_DEVICE";
        break;
    case IRP_MN_QUERY_DEVICE_RELATIONS:
        requeststring = "IRP_MN_QUERY_DEVICE_RELATIONS";
        break;

    case IRP_MN_QUERY_INTERFACE:
        //
        // This is the request we actually have to process.
        //
        requeststring = "IRP_MN_QUERY_INTERFACE";
        Status = RootHubPreProcessQueryInterface(Device, Irp);
        return Status;

    case IRP_MN_QUERY_CAPABILITIES:   
        requeststring = "IRP_MN_QUERY_CAPABILITIES";
        break;     
    case IRP_MN_QUERY_RESOURCES: 
        requeststring = "IRP_MN_QUERY_RESOURCES";
        break;          
    case IRP_MN_QUERY_RESOURCE_REQUIREMENTS:
        requeststring = "IRP_MN_QUERY_RESOURCE_REQUIREMENTS";
        break;
    case IRP_MN_QUERY_DEVICE_TEXT:
        requeststring = "IRP_MN_QUERY_DEVICE_TEXT";
        break;
    case IRP_MN_FILTER_RESOURCE_REQUIREMENTS:   
        requeststring = "IRP_MN_FILTER_RESOURCE_REQUIREMENTS";
        break;     
    case IRP_MN_READ_CONFIG:
        requeststring = "IRP_MN_READ_CONFIG";
        break;
    case IRP_MN_WRITE_CONFIG: 
        requeststring = "IRP_MN_WRITE_CONFIG";
        break;             
    case IRP_MN_EJECT: 
        traceLevel = TRACE_LEVEL_WARNING; 
        requeststring = "IRP_MN_EJECT";
        break;                   
    case IRP_MN_SET_LOCK:   
        requeststring = "IRP_MN_SET_LOCK";
        break;               
    case IRP_MN_QUERY_ID:  
        requeststring = "IRP_MN_QUERY_ID";
        break;                
    case IRP_MN_QUERY_PNP_DEVICE_STATE:  
        requeststring = "IRP_MN_QUERY_PNP_DEVICE_STATE";
        break;  
    case IRP_MN_QUERY_BUS_INFORMATION: 
        requeststring = "IRP_MN_QUERY_BUS_INFORMATION";
        break;    
    case IRP_MN_DEVICE_USAGE_NOTIFICATION: 
        requeststring = "IRP_MN_DEVICE_USAGE_NOTIFICATION";
        break;
    case IRP_MN_SURPRISE_REMOVAL:
        traceLevel = TRACE_LEVEL_INFORMATION;
        requeststring = "IRP_MN_SURPRISE_REMOVAL";
        break;
    default:
        break;
    };

    TraceEvents(traceLevel, TRACE_DEVICE,
        __FUNCTION__": WdfDevice %p %s (%x)\n",
        Device,
        requeststring,
        IoStack->MinorFunction);
    
    IoSkipCurrentIrpStackLocation(Irp);
    Status = WdfDeviceWdmDispatchPreprocessedIrp(Device, Irp);
    return Status;
}

NTSTATUS
RootHubPreProcessCreateIrp(
    IN WDFDEVICE Device,
    IN PIRP Irp)
{

    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_DEVICE,
        __FUNCTION__"\n");    
    
    IoSkipCurrentIrpStackLocation(Irp);
    NTSTATUS Status = WdfDeviceWdmDispatchPreprocessedIrp(Device, Irp);
    return Status;
}



/**
 * Transition to fully powered state.
 * @returns NTSTATUS value indicating success or failure. 
 */
NTSTATUS
HubEvtDeviceD0Entry(
    IN  WDFDEVICE Device,
    IN  WDF_POWER_DEVICE_STATE PreviousState)
{
    PUSB_HUB_PDO_CONTEXT hubContext = DeviceGetHubPdoContext(Device);

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_DEVICE,
        __FUNCTION__": Device %p PreviousState %s\n",
        hubContext->WdfDevice,
        DbgDevicePowerString(PreviousState));

    NTSTATUS status;
    PUSB_FDO_CONTEXT fdoContext = DeviceGetFdoContext(hubContext->Parent);

    status = WdfDeviceRetrieveDeviceInterfaceString(Device,
        &GUID_DEVINTERFACE_USB_HUB,
        NULL,
        fdoContext->hubsymlink);
    if (!NT_SUCCESS(status)) 
    {
        TraceEvents(TRACE_LEVEL_ERROR, TRACE_DEVICE,
            __FUNCTION__": %s WdfDeviceRetrieveDeviceInterfaceString for device %p error %x\n",
            fdoContext->FrontEndPath,
            Device,
            status);
    }    
    if (fdoContext->PortConnected)
    {
        hubContext->PortFeatureStatus |= USB_PORT_STATUS_CONNECT;
        hubContext->PortFeatureChange |= USB_PORT_STATUS_CONNECT;
        HubCheckStatusChange(hubContext);
    }

    return STATUS_SUCCESS;
}

/**
 * Transition out of fully powered state.
 * @returns NTSTATUS value indicating success or failure. 
 */
NTSTATUS
HubEvtDeviceD0Exit(
    IN  WDFDEVICE Device,
    IN  WDF_POWER_DEVICE_STATE TargetState)
{ 
    PUSB_HUB_PDO_CONTEXT hubContext = DeviceGetHubPdoContext(Device);

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_DEVICE,
        __FUNCTION__": Device %p TargetState %s\n",
        hubContext->WdfDevice,
        DbgDevicePowerString(TargetState));

    hubContext->PortFeatureStatus &= ~USB_PORT_STATUS_CONNECT;
    hubContext->PortFeatureChange |= USB_PORT_STATUS_CONNECT;
    HubCheckStatusChange(hubContext);

    return STATUS_SUCCESS;
}

/**
 * @brief callback on surprise removal pnp state transition.
 *
 * @param[in] Device. The handle for the hub pdo device.
 */
VOID HubEvtDeviceSurpriseRemoval(
  IN  WDFDEVICE Device)
{
    PUSB_HUB_PDO_CONTEXT hubContext = DeviceGetHubPdoContext(Device);

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_DEVICE,
        __FUNCTION__": Device %p unplugged\n",
        hubContext->WdfDevice);

    hubContext->PortFeatureStatus &= ~USB_PORT_STATUS_CONNECT;
    hubContext->PortFeatureChange |= USB_PORT_STATUS_CONNECT;
    HubCheckStatusChange(hubContext);
}

NTSTATUS
HubEvtDeviceQueryRemove (
  IN  WDFDEVICE Device)
{
    PUSB_HUB_PDO_CONTEXT hubContext = DeviceGetHubPdoContext(Device);  

    // Leave as a warning, devices should be surprise removed with no queries.
    TraceEvents(TRACE_LEVEL_WARNING, TRACE_DEVICE,
        __FUNCTION__": Device %p\n",
        hubContext->WdfDevice);
    return STATUS_SUCCESS;
}

/**
 * @brief initialize the IO queues for the hub.
 *
 * @param[in] Device. The handle for the hub pdo device.
 */
NTSTATUS
HubQueueInitialize(
    _In_ WDFDEVICE Device)
{
    WDFQUEUE queue;
    NTSTATUS status;
    WDF_IO_QUEUE_CONFIG    queueConfig;
    PUSB_HUB_PDO_CONTEXT   hubContext = DeviceGetHubPdoContext(Device);    
    PUSB_FDO_CONTEXT fdoContext = DeviceGetFdoContext(hubContext->Parent);

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_QUEUE,
        __FUNCTION__": %s device %p port connected %d\n",
        fdoContext->FrontEndPath,
        hubContext->WdfDevice,
        fdoContext->PortConnected);
    //
    // The default queue only processes IOCTL requests. These requests are targeted at the
    // HUB device and never go across the ringbuffer interface.
    //
    // The PdoUrbQueue handles InternalDeviceControl (URB) requests routed from the
    // child PDO. All it does is forward them to its parent.
    //
    WDF_IO_QUEUE_CONFIG_INIT_DEFAULT_QUEUE(
         &queueConfig,
        WdfIoQueueDispatchParallel); // this could probably be serial.

    queueConfig.EvtIoDeviceControl = HubEvtIoDeviceControl;
    queueConfig.EvtIoStop = HubEvtIoStop;

    status = WdfIoQueueCreate(
                 Device,
                 &queueConfig,
                 WDF_NO_OBJECT_ATTRIBUTES,
                 &queue);

    if( !NT_SUCCESS(status) ) 
    {
        TraceEvents(TRACE_LEVEL_ERROR, TRACE_QUEUE,
            __FUNCTION__": device %p Default queue WdfIoQueueCreate failed %x", 
            hubContext->WdfDevice,
            status);
        return status;
    }
    //
    // The URB queue
    //    
    WDF_IO_QUEUE_CONFIG_INIT(
        &queueConfig,
        WdfIoQueueDispatchParallel);

    queueConfig.EvtIoInternalDeviceControl = HubEvtIoInternalDeviceControl;
    queueConfig.EvtIoStop = HubEvtIoStop;
    status = WdfIoQueueCreate(Device,
        &queueConfig,
        WDF_NO_OBJECT_ATTRIBUTES,
        &hubContext->UrbQueue);

    if( !NT_SUCCESS(status) ) 
    {
        TraceEvents(TRACE_LEVEL_ERROR, TRACE_QUEUE,
            __FUNCTION__": device %p URB queue WdfIoQueueCreate failed %x", 
            hubContext->WdfDevice,
            status);
        return status;
    }

    status = WdfDeviceConfigureRequestDispatching(Device,
        hubContext->UrbQueue,
        WdfRequestTypeDeviceControlInternal);

    if( !NT_SUCCESS(status) ) 
    {
        TraceEvents(TRACE_LEVEL_ERROR, TRACE_QUEUE,
            __FUNCTION__": device %p URB queue WdfDeviceConfigureRequestDispatching failed %x", 
            hubContext->WdfDevice,
            status);
        return status;
    }    
    //
    // The status change queue
    //    
    WDF_IO_QUEUE_CONFIG_INIT(
        &queueConfig,
        WdfIoQueueDispatchManual); // manual dispatch

    queueConfig.EvtIoStop = HubStatusChangeEvtIoStop;

    status = WdfIoQueueCreate(Device,
        &queueConfig,
        WDF_NO_OBJECT_ATTRIBUTES,
        &hubContext->StatusChangeQueue);

    if( !NT_SUCCESS(status) ) 
    {
        TraceEvents(TRACE_LEVEL_ERROR, TRACE_QUEUE,
            __FUNCTION__": device %p StatusChange queue WdfIoQueueCreate failed %x", 
            hubContext->WdfDevice,
            status);
        return status;
    }
    hubContext->ReportedStatus = 0;
    if (fdoContext->PortConnected)
    {
        hubContext->PortFeatureStatus = USB_PORT_STATUS_CONNECT | USB_PORT_STATUS_HIGH_SPEED;
    }
    else
    {
        hubContext->PortFeatureStatus = 0;
    }
    hubContext->PortFeatureChange = USB_PORT_STATUS_CONNECT;

    return status;
}

USB_HUB_DESCRIPTOR RootHubDescriptor = 
{ 
    0x09,  // bDescLength
    0x29,  // bDescriptorType
    0x01,  // bNbrPorts
    0x1b,  // wHubCharacteristics ? or 0x01
    0x00,  // bPwrOn2PwrGood
    0x00,  // bHubContrCurrent
    0x00,  // DeviceRemovable
};

/**
 * @brief processes an IOCTL_USB_GET_NODE_INFORMATION request.
 * The IOCTL_USB_GET_NODE_INFORMATION I/O control request is used with 
 * the USB_NODE_INFORMATION structure to retrieve information about a 
 * parent device.
 * IOCTL_USB_GET_NODE_INFORMATION is a user-mode I/O control request. 
 * This request targets the USB hub device (GUID_DEVINTERFACE_USB_HUB).
 *
 * @param[in] fdoContext. The context object for the hub pdo device.
 * @param[in] Request. The handle to the IO Request.
 * @param[in] OutputBufferLength. The length in bytes of the output buffer.
 */
VOID
ProcessHubGetNodeInformation(
    IN PUSB_HUB_PDO_CONTEXT hubContext,
    IN WDFREQUEST Request,
    IN size_t)
{    
    NTSTATUS Status = STATUS_SUCCESS;
    ULONG_PTR Information = 0;
    PUSB_NODE_INFORMATION nodeInfo;
    PUSB_FDO_CONTEXT fdoContext = DeviceGetFdoContext(hubContext->Parent);
    
    Status = WdfRequestRetrieveOutputBuffer(Request,
            sizeof(USB_NODE_INFORMATION ),
            (PVOID *) &nodeInfo,
            NULL);    

    if (!NT_SUCCESS(Status))
    {
        Status = STATUS_BUFFER_TOO_SMALL;
    }
    else
    {
        nodeInfo->NodeType = UsbHub;
        nodeInfo->u.HubInformation.HubIsBusPowered = 0;
        RtlCopyMemory(&nodeInfo->u.HubInformation.HubDescriptor,
            &hubContext->HubConfig.HubDescriptor,
            sizeof(USB_HUB_DESCRIPTOR));
        Information = sizeof(USB_NODE_INFORMATION);
    }
    
    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
        __FUNCTION__": %s completing request with status %x\n",
        fdoContext->FrontEndPath,
        Status);

    WdfRequestCompleteWithInformation(Request, Status, Information);
}


/**
 * @brief processes an IOCTL_USB_GET_NODE_CONNECTION_INFORMATION or
 * IOCTL_USB_GET_NODE_CONNECTION_INFORMATION_EX request.
 *
 * The IOCTL_USB_GET_NODE_CONNECTION_INFORMATION request retrieves 
 * information about the indicated USB port and the device that is attached 
 * to the port, if there is one.
 *
 * Client drivers must send this IOCTL at an IRQL of PASSIVE_LEVEL.
 * IOCTL_USB_GET_NODE_CONNECTION_INFORMATION is a user-mode I/O control request. 
 * This request targets the USB hub device (GUID_DEVINTERFACE_USB_HUB).
 * 
 *
 * @param[in] fdoContext. The context object for the parent fdo device.
 * @param[in] Request. The handle to the IO Request.
 * @param[in] OutputBufferLength. The length in bytes of the output buffer.
 */
VOID
ProcessHubGetNodeConnectionInformation(
    IN PUSB_FDO_CONTEXT fdoContext,
    IN WDFREQUEST Request,
    IN size_t OutputBufferLength,
    IN BOOLEAN extended)
{ 
    ULONG_PTR Information = 0;
    PUSB_NODE_CONNECTION_INFORMATION connectionInfo;
    PUSB_NODE_CONNECTION_INFORMATION_EX connectionInfoEx;

    size_t minSizeNeeded = extended ? sizeof(USB_NODE_CONNECTION_INFORMATION_EX) : 
        sizeof(USB_NODE_CONNECTION_INFORMATION);

    ULONG pipesRequested = 0;
    
    NTSTATUS Status = WdfRequestRetrieveOutputBuffer(Request,
            minSizeNeeded,
            (PVOID *) &connectionInfo,
            NULL);    

    if (!NT_SUCCESS(Status))
    {
        Status = STATUS_BUFFER_TOO_SMALL;
    }
    else
    {
        connectionInfoEx = (PUSB_NODE_CONNECTION_INFORMATION_EX) connectionInfo;

        if (connectionInfo->ConnectionIndex != 1) // one based index, we only have one port.
        {
            Status = STATUS_INVALID_PARAMETER;
        }
        else
        {
            //
            // add in the number of pipes needed.
            XXX_TODO ("Incomplete processing of pipe info");
            //
            pipesRequested = (ULONG) ((OutputBufferLength - minSizeNeeded)/sizeof(USB_PIPE_INFO));
            pipesRequested = min(pipesRequested, fdoContext->NumEndpoints);

            minSizeNeeded += (sizeof(USB_PIPE_INFO) * pipesRequested);

            connectionInfo->DeviceDescriptor = fdoContext->DeviceDescriptor;
            connectionInfo->CurrentConfigurationValue = 0;
            connectionInfo->DeviceIsHub = FALSE;
            connectionInfo->DeviceAddress = 0; // @TODO pass through address
            connectionInfo->NumberOfOpenPipes = fdoContext->NumEndpoints;
            connectionInfo->ConnectionStatus = DeviceConnected;
            Information = minSizeNeeded;

            if (fdoContext->ConfigurationDescriptor && 
                (fdoContext->NumInterfaces > 0))
            {
                connectionInfo->CurrentConfigurationValue = 
                    fdoContext->ConfigurationDescriptor->bConfigurationValue;
                if (fdoContext->NumEndpoints)
                {
                    connectionInfo->NumberOfOpenPipes = fdoContext->NumEndpoints; // XXX is this right?
                    for (ULONG Index = 0;
                        Index < pipesRequested;
                        Index++)
                    {
                        connectionInfo->PipeList[Index].EndpointDescriptor = 
                            *fdoContext->PipeDescriptors[Index].endpoint;
                    }
                }
            }
            if (extended)
            {
                connectionInfoEx->Speed = (UCHAR) fdoContext->DeviceSpeed;
            }
            else
            {
                connectionInfo->LowSpeed = FALSE;
            }
        }
    }

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
        __FUNCTION__": %s length %d needed %d pipes %d requested %d request status %x\n",
        fdoContext->FrontEndPath,
        OutputBufferLength,
        minSizeNeeded,
        fdoContext->NumEndpoints,
        pipesRequested,
        Status);

    WdfRequestCompleteWithInformation(Request, Status, Information);
}

 /**
 * @brief processes an IOCTL_USB_GET_HUB_CAPABILITIES or 
 * IOCTL_USB_GET_HUB_CAPABILITIES_EX request.
 *
 * The IOCTL_USB_GET_HUB_CAPABILITIES I/O control request retrieves the 
 * capabilities of a USB hub.
 * Note  This request is replaced by IOCTL_USB_GET_HUB_CAPABILITIES_EX in Windows Vista.
 * IOCTL_USB_GET_HUB_CAPABILITIES is a user-mode I/O control request. 
 * This request targets the USB hub device (GUID_DEVINTERFACE_USB_HUB).
 * 
 *
 * @param[in] fdoContext. The context object for the parent fdo device.
 * @param[in] Request. The handle to the IO Request.
 * @param[in] OutputBufferLength. The length in bytes of the output buffer.
 */
VOID
ProcessHubGetCapabilities(
    IN PUSB_FDO_CONTEXT fdoContext,
    IN WDFREQUEST Request,
    IN size_t OutputBufferLength,
    IN BOOLEAN extended)
{ 
    ULONG_PTR Information = 0;
    PUSB_HUB_CAPABILITIES connectionInfo;
    PUSB_HUB_CAPABILITIES_EX connectionInfoEx;

    size_t minSizeNeeded = extended ? sizeof(USB_HUB_CAPABILITIES_EX) : 
        sizeof(USB_HUB_CAPABILITIES);

    ULONG pipesRequested = 0;
    
    NTSTATUS Status = WdfRequestRetrieveOutputBuffer(Request,
            minSizeNeeded,
            (PVOID *) &connectionInfo,
            NULL);

    if (!NT_SUCCESS(Status))
    {
        Status = STATUS_BUFFER_TOO_SMALL;
    }
    else
    {
        Information = minSizeNeeded;
        if (extended)
        {
            connectionInfoEx = (PUSB_HUB_CAPABILITIES_EX) connectionInfo;

            connectionInfoEx->CapabilityFlags.HubIsHighSpeedCapable = TRUE;
            connectionInfoEx->CapabilityFlags.HubIsHighSpeed = TRUE;
            // XXX
            connectionInfoEx->CapabilityFlags.HubIsMultiTt = TRUE; //??
            connectionInfoEx->CapabilityFlags.HubIsMultiTtCapable = TRUE; //?

            connectionInfoEx->CapabilityFlags.HubIsArmedWakeOnConnect = FALSE; // @TODO power management
            connectionInfoEx->CapabilityFlags.HubIsBusPowered = FALSE;
            connectionInfoEx->CapabilityFlags.HubIsRoot = TRUE;
            Status = STATUS_UNSUCCESSFUL;
        }
        else
        {
            connectionInfo->HubIs2xCapable = TRUE;
        }
    }
    
    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
        __FUNCTION__": %s length %d needed %d pipes %d requested %d request status %x\n",
        fdoContext->FrontEndPath,
        OutputBufferLength,
        minSizeNeeded,
        fdoContext->NumEndpoints,
        pipesRequested,
        Status);

    WdfRequestCompleteWithInformation(Request, Status, Information);
}

/**
 * @brief processes an IOCTL_USB_GET_DESCRIPTOR_FROM_NODE_CONNECTION.
 *
 * The IOCTL_USB_GET_DESCRIPTOR_FROM_NODE_CONNECTION I/O control request 
 * retrieves one or more descriptors for the device that is associated with 
 * the indicated port index.
 * IOCTL_USB_GET_DESCRIPTOR_FROM_NODE_CONNECTION is a user-mode I/O control 
 * request. This request targets the USB hub device 
 * (GUID_DEVINTERFACE_USB_HUB).
 *
 * @param[in] fdoContext. The context object for the parent fdo device.
 * @param[in] Request. The handle to the IO Request.
 * @param[in] OutputBufferLength. The length in bytes of the output buffer.
 */
VOID
ProcessHubGetDescriptorFromNodeConnection(
    IN PUSB_FDO_CONTEXT fdoContext,
    IN WDFREQUEST Request,
    IN size_t OutputBufferLength)
{
    PUSB_DESCRIPTOR_REQUEST descriptorRequest;
    ULONG_PTR Information = 0;

    NTSTATUS Status = WdfRequestRetrieveOutputBuffer(Request,
            sizeof(USB_DESCRIPTOR_REQUEST),
            (PVOID *) &descriptorRequest,
            NULL);

    if (NT_SUCCESS(Status))
    {
        ULONG dataLength = (ULONG) (OutputBufferLength - sizeof(USB_DESCRIPTOR_REQUEST));
        Status = ProcessGetDescriptorFromNode(fdoContext,
            descriptorRequest,
            &dataLength);

        if (NT_SUCCESS(Status))
        {
            Information = dataLength + sizeof(USB_DESCRIPTOR_REQUEST);
        }
    }
        
    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
        __FUNCTION__": %s completing request with status %x\n",
        fdoContext->FrontEndPath,
        Status);

    WdfRequestCompleteWithInformation(Request, Status, Information);
}

/**
 * @brief processes an IOCTL_USB_HUB_CYCLE_PORT.
 *
 * The IOCTL_USB_HUB_CYCLE_PORT I/O control request power-cycles the port 
 * that is associated with the PDO that receives the request.
 * Note  IOCTL_USB_HUB_CYCLE_PORT has been deprecated in Windows Vista 
 * and later operating systems. Do not use.
 * IOCTL_USB_HUB_CYCLE_PORT is a user-mode I/O control request. 
 * This request targets the USB hub device (GUID_DEVINTERFACE_USB_HUB).
 *
 * @param[in] hubContext. The context object for the hub pdo device.
 * @param[in] Request. The handle to the IO Request.
 * @param[in] OutputBufferLength. The length in bytes of the output buffer.
 *
 * @todo after a cycle port EVERYTHING about the attached usb device can change,
 * including device descriptor, config descriptor etc. 
 */
VOID
ProcessHubCyclePort(
    IN PUSB_HUB_PDO_CONTEXT hubContext,
    IN WDFREQUEST Request,
    IN size_t OutputBufferLength)
{
    PUSB_CYCLE_PORT_PARAMS cyclePort;
    ULONG_PTR Information = 0;
    UNREFERENCED_PARAMETER(OutputBufferLength);

    NTSTATUS Status = WdfRequestRetrieveOutputBuffer(Request,
            sizeof(USB_CYCLE_PORT_PARAMS),
            (PVOID *) &cyclePort,
            NULL);

    if (NT_SUCCESS(Status))
    {
        //
        // @TODO we only support a single port per root hub.
        // so ConnectionIndex is always "1".
        //
        if (cyclePort->ConnectionIndex == 1)
        {
            //
            // create a urb and send it to the controller fdo and wait for it to
            // complete, then complete this request.
            //
            WDFIOTARGET IoTarget;

            Status = WdfIoTargetCreate(hubContext->WdfDevice, // our hub device.
                WDF_NO_OBJECT_ATTRIBUTES,
                &IoTarget);
            if (NT_SUCCESS(Status))
            {
                WDF_IO_TARGET_OPEN_PARAMS Params;
                WDF_IO_TARGET_OPEN_PARAMS_INIT_EXISTING_DEVICE(&Params,
                    WdfDeviceWdmGetDeviceObject(hubContext->Parent) );
                Status = WdfIoTargetOpen(IoTarget, &Params);
                if (NT_SUCCESS(Status))
                {
                    //
                    // could optionally specify a timeout.
                    //
                    WDF_REQUEST_SEND_OPTIONS options;
                    WDF_REQUEST_SEND_OPTIONS_INIT(&options, WDF_REQUEST_SEND_OPTION_SYNCHRONOUS);

                    Status = WdfIoTargetSendInternalIoctlOthersSynchronously(
                        IoTarget,
                        NULL, //UrbRequest,
                        IOCTL_INTERNAL_USB_CYCLE_PORT,
                        NULL,
                        NULL,
                        NULL,
                        &options,
                        NULL);

                    if (!NT_SUCCESS(Status))
                    {
                        TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
                            __FUNCTION__": device %p WdfIoTargetSendInternalIoctlSynchronously error %x\n",
                            hubContext->WdfDevice,
                            Status);
                    }
                }
                else
                {
                    TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
                        __FUNCTION__": device %p WdfIoTargetOpen error %x\n",
                        hubContext->WdfDevice,
                        Status);
                }
                WdfObjectDelete(IoTarget);
            }
            else
            {
                TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
                    __FUNCTION__": device %p WdfIoTargetCreate error %x\n",
                    hubContext->WdfDevice,
                    Status);
            }            
            
        }
        else
        {
            cyclePort->StatusReturned = (ULONG) USBD_STATUS_INVALID_PARAMETER;
            Status = STATUS_UNSUCCESSFUL;
        }
    }
        
    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
        __FUNCTION__": %s completing request with status %x\n",
        GetFdoContextFromHubDevice(hubContext->WdfDevice)->FrontEndPath,
        Status);

    WdfRequestCompleteWithInformation(Request, Status, Information);
}
 
//
// no idea what this is.
// input is 16 bytes,
// output is 20 bytes.
//
#define IOCTL_WIN8_ACPI_ENUM_CHILDREN 0x32C020
 VOID
 ProcessHubAcpiEnumChildren(
    IN PUSB_HUB_PDO_CONTEXT hubContext,
    IN WDFREQUEST Request,
    IN size_t OutputBufferLength)
 {
     PUCHAR buffer;
     ULONG_PTR Information = 0;
     UNREFERENCED_PARAMETER(OutputBufferLength);
     NTSTATUS Status = WdfRequestRetrieveOutputBuffer(Request,
         20,
         (PVOID *) &buffer,
         NULL);

     if (NT_SUCCESS(Status))
     {
         RtlZeroMemory(buffer, 20);
         Information = 20;
     }
        
    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
        __FUNCTION__": %s completing request with status %x\n",
        GetFdoContextFromHubDevice(hubContext->WdfDevice)->FrontEndPath,
        Status);

    WdfRequestCompleteWithInformation(Request, Status, Information);

 }

 /**
 * @brief process IOCTL requests.
 * IOCTLs targeted at the root hub are processed here.
 *
 * @param[in] Queue the IOCTL queue.
 * @param[in] Request the IOCTL request handle.
 * @param[in] OutputBufferLength size of the IOCTL output buffer.
 * @param[in] InputBufferLength size of the IOCTL input buffer
 * @param[in] IoControlCode the IOCTL function code.
 */
VOID
HubEvtIoDeviceControl(
    _In_ WDFQUEUE Queue,
    _In_ WDFREQUEST Request,
    _In_ size_t OutputBufferLength,
    _In_ size_t InputBufferLength,
    _In_ ULONG IoControlCode
    )
{
    UNREFERENCED_PARAMETER(InputBufferLength);
    ULONG Function = IoGetFunctionCodeFromCtlCode(IoControlCode);
    PUSB_HUB_PDO_CONTEXT hubContext = DeviceGetHubPdoContext(WdfIoQueueGetDevice(Queue)); 
    PUSB_FDO_CONTEXT fdoContext = DeviceGetFdoContext(hubContext->Parent);

    switch (IoControlCode) 
    {
    case IOCTL_USB_GET_NODE_INFORMATION: // 258
        // must support
        ProcessHubGetNodeInformation(hubContext, Request, OutputBufferLength);
        Request = NULL;
        break;

    case IOCTL_USB_GET_NODE_CONNECTION_INFORMATION: // 259
        ProcessHubGetNodeConnectionInformation(fdoContext, Request, OutputBufferLength, FALSE);
        Request = NULL;
        break;

    case IOCTL_USB_GET_NODE_CONNECTION_INFORMATION_EX: // 274
        ProcessHubGetNodeConnectionInformation(fdoContext, Request, OutputBufferLength, TRUE);
        Request = NULL;
        break;

    case IOCTL_USB_GET_DESCRIPTOR_FROM_NODE_CONNECTION: //260
        ProcessHubGetDescriptorFromNodeConnection(fdoContext, Request, OutputBufferLength);
        Request = NULL;
        break;            

    case IOCTL_USB_GET_NODE_CONNECTION_DRIVERKEY_NAME: // 264
        //
        // this is really a wrapper around IoGetDeviceProperty type
        // DevicePropertyDriverKeyName 
        // 
        ProcessDriverKeyNameRequest(fdoContext, Request, OutputBufferLength);
        Request = NULL;
        break; 

    case IOCTL_USB_GET_HUB_CAPABILITIES:                  // 271
        ProcessHubGetCapabilities(fdoContext, Request, OutputBufferLength, FALSE);
        Request = NULL;
        break; 

    case IOCTL_USB_HUB_CYCLE_PORT: // 273
        ProcessHubCyclePort(hubContext, Request, OutputBufferLength);
        Request = NULL;
        break;        

    case IOCTL_USB_GET_HUB_CAPABILITIES_EX:               // 276 Used by usbview.
        ProcessHubGetCapabilities(fdoContext, Request, OutputBufferLength, TRUE);
        Request = NULL;
        break;

    case IOCTL_WIN8_ACPI_ENUM_CHILDREN:
        // Don't support this for now.
        // ProcessHubAcpiEnumChildren(hubContext,Request, OutputBufferLength);
        // Request = NULL;
        break;

    case IOCTL_USB_GET_NODE_CONNECTION_NAME:              // 261
    case IOCTL_USB_GET_NODE_CONNECTION_ATTRIBUTES:        // 272
    case IOCTL_USB_DIAG_IGNORE_HUBS_ON:                   // 262
    case IOCTL_USB_DIAG_IGNORE_HUBS_OFF:                  // 263
    case IOCTL_USB_RESET_HUB:                             // 275
#if (NTDDI_VERSION >= NTDDI_WIN8)
    //
    //  These are used by usbview built for win8:
    //  USB_GET_HUB_INFORMATION_EX                  277
    //  USB_GET_PORT_CONNECTOR_PROPERTIES           278
    //  USB_GET_NODE_CONNECTION_INFORMATION_EX_V2   279
    //
    case IOCTL_USB_GET_HUB_INFORMATION_EX:                // 277
    case IOCTL_USB_GET_PORT_CONNECTOR_PROPERTIES:         // 278
    case IOCTL_USB_GET_NODE_CONNECTION_INFORMATION_EX_V2: // 279
#endif
    default:

        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": %s  HUB IOCTL %x (%d) %s unsupported\n",
            fdoContext->FrontEndPath,
            IoControlCode,
            Function,
            UsbIoctlToString(IoControlCode));
        break;
    }

    if (Request)
    {
        WdfRequestComplete(Request, STATUS_UNSUCCESSFUL);
    }

    return;
}

/**
 * @brief wraps WdfRequestForwardToParentDeviceIoQueue.
 *
 */
VOID
ForwardToParent(
    IN WDFREQUEST Request,
    IN PUSB_HUB_PDO_CONTEXT hubContext)
{
    WDF_REQUEST_FORWARD_OPTIONS forwardOptions;
    WDF_REQUEST_FORWARD_OPTIONS_INIT(&forwardOptions);

    NTSTATUS Status = WdfRequestForwardToParentDeviceIoQueue(
        Request,
        hubContext->ParentQueue,
        &forwardOptions);

    if (!NT_SUCCESS(Status))
    {
        char * errorString = "Unexpected error code";
        char * infoString = "";

        switch (Status)
        {
        case STATUS_INFO_LENGTH_MISMATCH:
            errorString = "STATUS_INFO_LENGTH_MISMATCH";
            infoString = "size of forwardOptions invalid";
            break;

        case STATUS_INVALID_PARAMETER:
            errorString = "STATUS_INVALID_PARAMETER";
            infoString = "forwardOptions invalid member";
            break;

        case STATUS_INVALID_DEVICE_REQUEST:
            errorString = "STATUS_INVALID_DEVICE_REQUEST";
            infoString = "assorted queue processing malfunctions";
            break;

        case STATUS_WDF_BUSY:
            errorString = "STATUS_WDF_BUSY";
            infoString = "parent not accepting new requests";
            XXX_TODO("Handle WDF_BUSY condition in device pdo");
            break;

        default:
            break;
        };

        TraceEvents(TRACE_LEVEL_ERROR, TRACE_QUEUE,
            __FUNCTION__": %s Device %p Request %p ParentQueue %p Parent Device %p "
            "WdfRequestForwardToParentDeviceIoQueue error %x %s %s\n",
            GetFdoContextFromHubDevice(hubContext->WdfDevice)->FrontEndPath,
            hubContext->WdfDevice,
            Request,
            hubContext->ParentQueue,
            hubContext->Parent,
            Status,
            errorString,
            infoString);

        WdfRequestComplete(Request, STATUS_UNSUCCESSFUL);
    }

}

/**
* @brief return the HUB's USB Device Descriptor.
*
* @param[in] hubContext the context for this hub device.
* @oaram[in] Request the Request object.
* @param[in] Urb the URB provided by the Reauest.
* @param[in] buffer the data payload for the URB
* @param[in] packet the control and setup packet from the URB.
*
*/
VOID
HubGetDeviceDescriptor(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PUCHAR buffer,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    UNREFERENCED_PARAMETER(packet);
    ULONG bytesToCopy;
    NTSTATUS Status = STATUS_UNSUCCESSFUL;
    Urb->UrbHeader.Status = USBD_STATUS_INVALID_PARAMETER;

    if ((Urb->UrbControlDescriptorRequest.TransferBufferLength >= sizeof(USB_DEVICE_DESCRIPTOR)))
    {
        bytesToCopy = min(Urb->UrbControlDescriptorRequest.TransferBufferLength, 
            hubContext->HubConfig.DeviceDescriptor.bLength);

        RtlCopyMemory(buffer,
            (PVOID) &hubContext->HubConfig.DeviceDescriptor,
            bytesToCopy);

        Urb->UrbControlDescriptorRequest.TransferBufferLength = bytesToCopy;

        TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
            __FUNCTION__": device %p descriptor processed locally %d bytes returned\n",
            hubContext->WdfDevice,
            bytesToCopy);

        Status = STATUS_SUCCESS;
        Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
    }
    else
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": device %p TransferBufferLength too small (%d)\n",
            hubContext->WdfDevice,
            Urb->UrbControlDescriptorRequest.TransferBufferLength,
            sizeof(USB_DEVICE_DESCRIPTOR));
    }

    WdfRequestComplete(Request, Status);
}

/**
* @brief return the USB 2.0 HUB descriptor (Chapter 11 Hub Specification USB 2.0 spec.)
*
* @param[in] hubContext the context for this hub device.
* @param[in] Request the Request object.
* @param[in] Urb the URB provided by the Reauest.
* @param[in] buffer the data payload for the URB.
* @param[in] packet the control and setup packet from the URB.
*
*/
VOID
HubGetHubDescriptor(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PUCHAR buffer,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    UNREFERENCED_PARAMETER(packet);
    ULONG bytesToCopy;
    NTSTATUS Status = STATUS_UNSUCCESSFUL;
    Urb->UrbHeader.Status = USBD_STATUS_INVALID_PARAMETER;

    if ((Urb->UrbControlDescriptorRequest.TransferBufferLength >= sizeof(USB_HUB_DESCRIPTOR)))
    {
        bytesToCopy = min(Urb->UrbControlDescriptorRequest.TransferBufferLength, 
            hubContext->HubConfig.HubDescriptor.bDescriptorLength);

        RtlCopyMemory(buffer,
            (PVOID) &hubContext->HubConfig.HubDescriptor,
            bytesToCopy);

        Urb->UrbControlDescriptorRequest.TransferBufferLength = bytesToCopy;

        TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
            __FUNCTION__": device %p descriptor processed locally %d bytes returned\n",
            hubContext->WdfDevice,
            bytesToCopy);

        Status = STATUS_SUCCESS;
        Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
    }
    else
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": device %p TransferBufferLength too small (%d)\n",
            hubContext->WdfDevice,
            Urb->UrbControlDescriptorRequest.TransferBufferLength,
            sizeof(USB_HUB_DESCRIPTOR));
    }

    WdfRequestComplete(Request, Status);
}

/**
 * @brief return the ENTIRE config, config_des + interface_desc(s) + endpoint_desc(s),
 * if there is room, else just the config_desc.
 * @note this is the config descriptor for the HUB USB device, not  for one of the
 * devices attached to its ports.
 *
 * @param[in] hubContext the context for this hub device.
 * @param[in] Request the Request object.
 * @param[in] Urb the URB provided by the Reauest.
 * @param[in] buffer the data payload for the URB.
 * @param[in] packet the control and setup packet from the URB.
 *
 */
VOID
HubGetConfigurationDescriptor(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PUCHAR buffer,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    UNREFERENCED_PARAMETER(packet);
    ULONG bytesToCopy;
    NTSTATUS Status = STATUS_UNSUCCESSFUL;
    Urb->UrbHeader.Status = USBD_STATUS_INVALID_PARAMETER;
    ULONG entireConfigSize = hubContext->HubConfig.ConfigurationDescriptor.wTotalLength;
    ULONG bytesCopied = 0;

    if ((Urb->UrbControlDescriptorRequest.TransferBufferLength >= sizeof(USB_CONFIGURATION_DESCRIPTOR)))
    {
        bytesToCopy = sizeof(USB_CONFIGURATION_DESCRIPTOR);
        RtlCopyMemory(buffer,
            &hubContext->HubConfig.ConfigurationDescriptor,
            bytesToCopy);
        buffer += bytesToCopy;
        bytesCopied += bytesToCopy;

        if (Urb->UrbControlDescriptorRequest.TransferBufferLength >= entireConfigSize)
        {
            bytesToCopy = sizeof(USB_INTERFACE_DESCRIPTOR);
            RtlCopyMemory(buffer,
                &hubContext->HubConfig.InterfaceDescriptor,
                bytesToCopy);
            buffer += bytesToCopy;
            bytesCopied += bytesToCopy;
            
            bytesToCopy = sizeof(USB_ENDPOINT_DESCRIPTOR);
            RtlCopyMemory(buffer,
                &hubContext->HubConfig.EndpointDescriptor,
                bytesToCopy);
            bytesCopied += bytesToCopy;
        }

        Urb->UrbControlDescriptorRequest.TransferBufferLength = bytesCopied;

        TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
            __FUNCTION__": Device %p config descriptor %d totalLength %d. %d bytes returned\n",
            hubContext->WdfDevice,
            Urb->UrbControlDescriptorRequest.Index,
            entireConfigSize,
            bytesCopied);

        Status = STATUS_SUCCESS;
        Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
    }
    else
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": Device %p TransferBufferLength %d too small (%d)\n",
            hubContext->WdfDevice,
            Urb->UrbControlDescriptorRequest.TransferBufferLength,
            sizeof(USB_CONFIGURATION_DESCRIPTOR));
    }

    WdfRequestComplete(Request, Status);
}
/**
* @brief returns a hub device endpoint descriptor.
*
* @param[in] hubContext the context for this hub device.
* @param[in] Request the Request object.
* @param[in] Urb the URB provided by the Reauest.
* @param[in] buffer the data payload for the URB.
* @param[in] packet the control and setup packet from the URB.
*
*/
VOID
HubGetEndpointDescriptor(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PUCHAR buffer,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    UNREFERENCED_PARAMETER(packet);
    ULONG bytesToCopy;
    NTSTATUS Status = STATUS_UNSUCCESSFUL;
    Urb->UrbHeader.Status = USBD_STATUS_INVALID_PARAMETER;

    if ((Urb->UrbControlDescriptorRequest.TransferBufferLength >= sizeof(USB_ENDPOINT_DESCRIPTOR)))
    {
        bytesToCopy = min(Urb->UrbControlDescriptorRequest.TransferBufferLength, 
            hubContext->HubConfig.EndpointDescriptor.bLength);

        RtlCopyMemory(buffer,
            &hubContext->HubConfig.EndpointDescriptor,
            bytesToCopy);            

        Urb->UrbControlDescriptorRequest.TransferBufferLength = bytesToCopy;

        TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
            __FUNCTION__": Device %p config descriptor %d processed locally %d bytes returned\n",
            hubContext->WdfDevice,
            Urb->UrbControlDescriptorRequest.Index,
            bytesToCopy);

        Status = STATUS_SUCCESS;
        Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
    }
    else
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": Device %p TransferBufferLength %d too small (%d)\n",
            hubContext->WdfDevice,
            Urb->UrbControlDescriptorRequest.TransferBufferLength,
            sizeof(USB_ENDPOINT_DESCRIPTOR));
    }

    WdfRequestComplete(Request, Status);
}

/**
* @brief returns a hub device interface descriptor.
*
* @param[in] hubContext the context for this hub device.
* @param[in] Request the Request object.
* @param[in] Urb the URB provided by the Reauest.
* @param[in] buffer the data payload for the URB.
* @param[in] packet the control and setup packet from the URB.
*
*/
VOID
HubGetInterfaceDescriptor(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PUCHAR buffer,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    UNREFERENCED_PARAMETER(packet);
    ULONG bytesToCopy;
    NTSTATUS Status = STATUS_UNSUCCESSFUL;
    Urb->UrbHeader.Status = USBD_STATUS_INVALID_PARAMETER;

    if ((Urb->UrbControlDescriptorRequest.TransferBufferLength >= sizeof(USB_INTERFACE_DESCRIPTOR)))
    {
        bytesToCopy = min(Urb->UrbControlDescriptorRequest.TransferBufferLength, 
            hubContext->HubConfig.InterfaceDescriptor.bLength);

        RtlCopyMemory(buffer,
            &hubContext->HubConfig.ConfigurationDescriptor,
            bytesToCopy);            

        Urb->UrbControlDescriptorRequest.TransferBufferLength = bytesToCopy;

        TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
            __FUNCTION__": Device %p config descriptor %d processed locally %d bytes returned\n",
            hubContext->WdfDevice,
            Urb->UrbControlDescriptorRequest.Index,
            bytesToCopy);

        Status = STATUS_SUCCESS;
        Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
    }
    else
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": Device %p TransferBufferLength %d too small (%d)\n",
            hubContext->WdfDevice,
            Urb->UrbControlDescriptorRequest.TransferBufferLength,
            sizeof(USB_INTERFACE_DESCRIPTOR));
    }

    WdfRequestComplete(Request, Status);
}

/**
* @brief returns a hub device string descriptor.
* @note there are no string descriptors implemented.
*
* @param[in] hubContext the context for this hub device.
* @param[in] Request the Request object.
* @param[in] Urb the URB provided by the Reauest.
* @param[in] buffer the data payload for the URB.
* @param[in] packet the control and setup packet from the URB.
*
*/
VOID
HubGetStringDescriptor(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PUCHAR buffer,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    UNREFERENCED_PARAMETER(packet);    
    UNREFERENCED_PARAMETER(buffer);
    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
        __FUNCTION__": Device %p Request %p URB %p\n",
        hubContext->WdfDevice,
        Request,
        Urb);
    // ULONG bytesToCopy;
    NTSTATUS Status = STATUS_UNSUCCESSFUL;
    Urb->UrbHeader.Status = USBD_STATUS_INVALID_PARAMETER;
    WdfRequestComplete(Request, Status);
}

/**
* @brief gets the current active configuration setting for the hub device.
*
* @param[in] hubContext the context for this hub device.
* @param[in] Request the Request object.
* @param[in] Urb the URB provided by the Reauest.
* @param[in] buffer the data payload for the URB.
* @param[in] packet the control and setup packet from the URB.
*
*/
VOID
HubGetConfiguration(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PUCHAR buffer,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    UNREFERENCED_PARAMETER(packet);

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
        __FUNCTION__": Device %p Request %p URB %p length %d\n",
        hubContext->WdfDevice,
        Request,
        Urb,
        Urb->UrbControlTransferEx.TransferBufferLength);

    if (Urb->UrbControlTransferEx.TransferBufferLength >= 1)
    {
        buffer[0] = 1;
        Urb->UrbControlDescriptorRequest.TransferBufferLength = 1;
        Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
        WdfRequestComplete(Request, STATUS_SUCCESS);
        return;
    }

    TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
        __FUNCTION__": request not processed\n");

    Urb->UrbHeader.Status = USBD_STATUS_INVALID_PARAMETER;
    WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
}

/**
 * @brief Process GET_STATUS request for HUB or PORT.
 * See USB2.0 11.24.2.6
 *
 * @param[in] hubContext the context for this hub device.
 * @oaram[in] Request the Request object.
 * @param[in] Urb the URB provided by the Reauest.
 * @param[in] buffer the data payload for the URB
 * @param[in] packet the control and setup packet from the URB.
 * 
 */
VOID
HubGetStatus(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PUCHAR buffer,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
        __FUNCTION__": Device %p Request %p URB %p length %d\n",
        hubContext->WdfDevice,
        Request,
        Urb,
        Urb->UrbControlTransferEx.TransferBufferLength);
    //
    // A standard GET_STATUS request
    //
    if (packet->Packet.bm.Request.Type == BMREQUEST_STANDARD)
    {
        if ((packet->Packet.bm.Request.Recipient == BMREQUEST_TO_DEVICE) &&
            (Urb->UrbControlTransferEx.TransferBufferLength >= 2))
        {
            // assume the rest of the packet is kosher.
            buffer[0] = 1;
            buffer[1] = 0;
            Urb->UrbControlDescriptorRequest.TransferBufferLength = 2;
            Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
            WdfRequestComplete(Request, STATUS_SUCCESS);
            return;
        }
    }
    else if ((packet->Packet.bm.Request.Type == BMREQUEST_CLASS) &&
        (Urb->UrbControlTransferEx.TransferBufferLength >= 4))
    {
        //
        // A Class Specific GET_STATUS request.
        //
        RtlZeroMemory(buffer, 4);
        if (packet->Packet.bm.Request.Recipient == BMREQUEST_TO_OTHER)
        {
            //
            // GetPortStatus. See 11.24.2.7
            //
            if (packet->Packet.wIndex.Value == 1)
            {
                //
                // see if our parent device thinks the port is connected.
                //
                PUSB_FDO_CONTEXT fdoContext = GetFdoContextFromHubDevice(hubContext->WdfDevice);
                if (fdoContext->PortConnected)
                {
                    if (!(hubContext->PortFeatureStatus & USB_PORT_STATUS_CONNECT))
                    {
                        //
                        // Don't transition from disconnected to connected
                        // based on fdoContext->PortConnected = TRUE
                        // This could be a hub directed cycle port.
                        //
                        if (!hubContext->HubDisconnectReported)
                        {
                            // only log this once per occurrence.

                            TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
                                __FUNCTION__": Device %p PortFeatureStatus is Disconnected\n",
                                hubContext->WdfDevice);
                            hubContext->HubDisconnectReported = TRUE;
                        }
                    }
                    else if (hubContext->HubDisconnectReported)
                    {
                        hubContext->HubDisconnectReported = FALSE;
                    }
                }
                else if (hubContext->PortFeatureStatus & USB_PORT_STATUS_CONNECT)
                {   
                        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
                            __FUNCTION__": Device %p PortConnected is FALSE PortFeatureStatus is connected\n",
                            hubContext->WdfDevice);                 
                    hubContext->PortFeatureStatus &= ~USB_PORT_STATUS_CONNECT;
                    hubContext->PortFeatureChange |= USB_PORT_STATUS_CONNECT; // @todo is this correct?
                }
                PUSB_20_PORT_STATUS portStatus = (PUSB_20_PORT_STATUS) buffer;
                PUSB_20_PORT_CHANGE portChange = (PUSB_20_PORT_CHANGE) &buffer[2];
                portStatus->AsUshort16 = hubContext->PortFeatureStatus;
                portChange->AsUshort16 = hubContext->PortFeatureChange;

                if (hubContext->PortFeatureStatus & USB_PORT_STATUS_RESET)
                {
                    hubContext->PortFeatureStatus &= ~USB_PORT_STATUS_RESET;
                    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
                        __FUNCTION__": Device %p clearing port feature reset status\n",
                        hubContext->WdfDevice);
                }
                ULONG level = hubContext->PortFeatureChange ? TRACE_LEVEL_WARNING : 
                    TRACE_LEVEL_VERBOSE;

                TraceEvents(level, TRACE_URB,
                    __FUNCTION__": Device %p Port Status %x Change %x\n",
                    hubContext->WdfDevice,
                    portStatus->AsUshort16,
                    portChange->AsUshort16);

            }
        }
        else if (packet->Packet.bm.Request.Recipient == BMREQUEST_TO_DEVICE)
        {
            //
            // GetHubStatus
            // Zeroes is good right? See 11.24.2.6
            //
        }
        Urb->UrbControlDescriptorRequest.TransferBufferLength = 4;
        Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
        WdfRequestComplete(Request, STATUS_SUCCESS);
        return;
    }

    TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
        __FUNCTION__": request not processed\n");

    Urb->UrbHeader.Status = USBD_STATUS_INVALID_PARAMETER;
    WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
}


PCHAR
DecodeFeature(USHORT value)
{
    PCHAR valueString = "Unknown Feature";

    switch (value)
    {
    case PORT_CONNECTION:
        valueString = "PORT_CONNECTION";
        break;
    case PORT_ENABLE:
        valueString = "PORT_ENABLE";
        break;
    case PORT_SUSPEND:
        valueString = "PORT_SUSPEND";
        break;
    case PORT_OVER_CURRENT:
        valueString = "PORT_OVER_CURRENT";
        break;
    case PORT_RESET:
        valueString = "PORT_RESET";
        break;
    case PORT_LINK_STATE:
        valueString = "PORT_LINK_STATE";
        break;
    case PORT_POWER:
        valueString = "PORT_POWER";
        break;
    case PORT_LOW_SPEED:
        valueString = "PORT_LOW_SPEED";
        break;
    case C_PORT_CONNECTION:
        valueString = "C_PORT_CONNECTION";
        break;
    case C_PORT_ENABLE:
        valueString = "C_PORT_ENABLE";
        break;
    case C_PORT_SUSPEND:
        valueString = "C_PORT_SUSPEND";
        break;
    case C_PORT_OVER_CURRENT:
        valueString = "C_PORT_OVER_CURRENT";
        break;
    case C_PORT_RESET:
        valueString = "C_PORT_RESET";
        break;
    case PORT_TEST:
        valueString = "PORT_TEST";
        break;
    case PORT_INDICATOR:
        valueString = "PORT_INDICATOR";
        break;
    case PORT_U1_TIMEOUT:
        valueString = "PORT_U1_TIMEOUT";
        break;
    case PORT_U2_TIMEOUT:
        valueString = "PORT_U2_TIMEOUT";
        break;
    case C_PORT_LINK_STATE:
        valueString = "C_PORT_LINK_STATE";
        break;
    case C_PORT_CONFIG_ERROR:
        valueString = "C_PORT_CONFIG_ERROR";
        break;
    case PORT_REMOTE_WAKE_MASK:
        valueString = "PORT_REMOTE_WAKE_MASK";
        break;
    case BH_PORT_RESET:
        valueString = "BH_PORT_RESET";
        break;
    case C_BH_PORT_RESET:
        valueString = "C_BH_PORT_RESET";
        break;
    case FORCE_LINKPM_ACCEPT:
        valueString = "FORCE_LINKPM_ACCEPT";
        break;
    }
    return valueString;
}

/** @brief process a SET_FEATURE request.
  *  See Figure 11-10. Downstream Facing Hub Port State Machine. USB2.0 Spec.
  *  Set HUB Feature 
  *  bmRequestType = 0x20
  *  bRequest = 0x03
  *  wValue - feature, see table below.
  *  wIndex = 0
  *  wLength = 0
  *
  *  Set PORT Feature.
  *  bmRequestType = 0x23
  *  bRequest = 0x03
  *  wValue - feature, see table below.
  *  wIndex = LSB = Port.
  *  wLength = 0
  *
  *
  *  Class Feature Selectors:
  *
    Feature             | Recp | Value
    --------------------|------|------
    C_HUB_LOCAL_POWER   | Hub  | 0
    C_HUB_OVER_CURRENT  | Hub  | 1
    PORT_CONNECTION     | Port | 0
    PORT_ENABLE         | Port | 1
    PORT_SUSPEND        | Port | 2
    PORT_OVER_CURRENT   | Port | 3
    PORT_RESET          | Port | 4
    PORT_POWER          | Port | 8
    PORT_LOW_SPEED      | Port | 9
    C_PORT_CONNECTION   | Port | 16
    C_PORT_ENABLE       | Port | 17
    C_PORT_SUSPEND      | Port | 18
    C_PORT_OVER_CURRENT | Port | 19
    C_PORT_RESET        | Port | 20
    PORT_TEST           | Port | 21
    PORT_INDICATOR      | Port | 22  
 *
 * @param[in] hubContext the context for this hub device.
 * @oaram[in] Request the Request object.
 * @param[in] Urb the URB provided by the Reauest.
 * @param[in] buffer the data payload for the URB
 * @param[in] packet the control and setup packet from the URB.
*/
VOID
HubSetFeature(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    PCHAR feature = DecodeFeature(packet->Packet.wValue.Value);
    BOOLEAN statusChanged = FALSE; 
    PUSB_FDO_CONTEXT fdoContext = GetFdoContextFromHubDevice(hubContext->WdfDevice);


    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
        __FUNCTION__": Device %p Request %p URB %p Feature %s %d wIndex %x\n",
        hubContext->WdfDevice,
        Request,
        Urb,
        feature,
        packet->Packet.wValue.Value,
        packet->Packet.wIndex.Value);

    if ((packet->Packet.bm.Request.Type == BMREQUEST_CLASS) &&
        (packet->Packet.bm.Request.Recipient == BMREQUEST_TO_OTHER))
    {
        switch (packet->Packet.wValue.Value)
        {
        case PORT_ENABLE:
            break;
        case PORT_RESET:
            // @todo actually reset the device here rather than on our own.
            hubContext->PortFeatureStatus |= USB_PORT_STATUS_ENABLE|USB_PORT_STATUS_RESET;
            hubContext->PortFeatureChange |= USB_PORT_STATUS_RESET;
            ResetDevice(fdoContext);
            statusChanged = TRUE;
            break;
        case PORT_SUSPEND:
            break;
        case PORT_POWER:
            hubContext->PortFeatureStatus |= USB_PORT_STATUS_POWER;
            break;
        case PORT_TEST:
            break;
        default:
            break;
        }   

        TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
            __FUNCTION__": Device %p Feature %s %d PortFeatureStatus %x PortFeatureChange: %x\n",
            hubContext->WdfDevice,
            feature,
            packet->Packet.wValue.Value,
            hubContext->PortFeatureStatus,
            hubContext->PortFeatureChange);
    }
    
    Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
    WdfRequestComplete(Request, STATUS_SUCCESS);
    if (statusChanged)
    {
        HubCheckStatusChange(hubContext);
    }
}

/**
 * @brief clear a feature set by HubSetFeature().
 *
 * @param[in] hubContext the context for this hub device.
 * @oaram[in] Request the Request object.
 * @param[in] Urb the URB provided by the Reauest.
 * @param[in] buffer the data payload for the URB
 * @param[in] packet the control and setup packet from the URB
 *
 */
VOID
HubClearFeature(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb,
    IN PWDF_USB_CONTROL_SETUP_PACKET packet)
{
    PCHAR feature = DecodeFeature(packet->Packet.wValue.Value);
    BOOLEAN statusChanged = FALSE;

    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
        __FUNCTION__": Device %p Request %p URB %p Feature %s %d wIndex %x\n",
        hubContext->WdfDevice,
        Request,
        Urb,
        feature,
        packet->Packet.wValue.Value,
        packet->Packet.wIndex.Value);

    if ((packet->Packet.bm.Request.Type == BMREQUEST_STANDARD) &&
        (packet->Packet.bm.Request.Recipient == BMREQUEST_TO_DEVICE))
    {
        // ClearHubFeature: only C_HUB_LOCAL_POWER and C_HUB_OVER_CURRENT
        // are supported. All hub clear feature operations are ignored!
        if (packet->Packet.wValue.Value == C_HUB_LOCAL_POWER)
        {
            feature = "C_HUB_LOCAL_POWER";
        }
        else if (packet->Packet.wValue.Value == C_HUB_OVER_CURRENT)
        {
            feature = "C_HUB_OVER_CURRENT";
        }
        else
        {
            // ignore bad hub features too.
            TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
                __FUNCTION__": Hub feature %s unsupported\n",
                feature);
        }

    }
    else if ((packet->Packet.bm.Request.Type == BMREQUEST_CLASS) &&
        (packet->Packet.bm.Request.Recipient == BMREQUEST_TO_OTHER))
    {
        // ClearPortFeature

        switch (packet->Packet.wValue.Value)
        {
        case PORT_ENABLE:
            hubContext->PortFeatureStatus &= ~USB_PORT_STATUS_ENABLE;
            hubContext->PortFeatureChange |= USB_PORT_STATUS_ENABLE;
            statusChanged = TRUE;
            break;
        case PORT_SUSPEND:
            break;
        case PORT_POWER:
            break;
        case PORT_TEST:
            break;
        case C_PORT_CONNECTION:
            hubContext->PortFeatureChange &= ~USB_PORT_STATUS_CONNECT;
            break;
        case C_PORT_ENABLE: // ? is this allowed ? (Sent by MUTT).
            hubContext->PortFeatureChange &= ~USB_PORT_STATUS_ENABLE;
            break;
        case C_PORT_RESET:
            hubContext->PortFeatureChange &= ~USB_PORT_STATUS_RESET;
        default:
            break;
        }

        TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
            __FUNCTION__": Device %p Feature %s %d PortFeatureChange: %x\n",
            hubContext->WdfDevice,
            feature,
            packet->Packet.wValue.Value,
            hubContext->PortFeatureChange);
    }
    
    Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
    WdfRequestComplete(Request, STATUS_SUCCESS);
    if (statusChanged)
    {
        HubCheckStatusChange(hubContext);
    }

}

/** @brief processes Hub specific requests sent via URB_FUNCTION_CONTROL_TRANSFER_EX.

    bRequest        | Value
    ---------------------
    GET_ STATUS     | 0
    CLEAR_ FEATURE  | 1
    RESERVED        | 2
    SET_ FEATURE    | 3
    RESERVED        | 4-5
    GET_DESCRIPTOR  | 6
    SET_DESCRIPTOR  | 7
    CLEAR_TT_BUFFER | 8
    RESET_TT        | 9
    GET_TT_STATE    | 10
    STOP_TT         | 11

 *
 * @param[in] hubContext the context for this hub device.
 * @oaram[in] Request the Request object.
 * @param[in] Urb the URB provided by the Reauest.
 * @param[in] buffer the data payload for the URB
 * @param[in] packet the control and setup packet from the URB
 *
 *
 */
VOID
HubControlTransferEx(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb)
{
    WDF_USB_CONTROL_SETUP_PACKET packet;
    RtlCopyMemory(packet.Generic.Bytes, Urb->UrbControlTransferEx.SetupPacket, 8);

    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
        __FUNCTION__": Device %p Request %p URB %p\n"
        " bmRequest Dir %d Type %d Recipient %d bRequest %x wValue %x wIndex %x\n",
        hubContext->WdfDevice,
        Request,
        Urb,
        packet.Packet.bm.Request.Dir,
        packet.Packet.bm.Request.Type,
        packet.Packet.bm.Request.Recipient,
        packet.Packet.bRequest,
        packet.Packet.wValue.Value,
        packet.Packet.wIndex.Value);

    PCHAR transferRequest = "Unknown";

    PUCHAR buffer = (PUCHAR) Urb->UrbControlTransferEx.TransferBuffer;
    if (Urb->UrbControlTransferEx.TransferBufferMDL)
    {
        buffer = (PUCHAR) MmGetSystemAddressForMdlSafe(Urb->UrbControlTransferEx.TransferBufferMDL,
            NormalPagePriority );
    }
    if (buffer == NULL)
    {
        // this can be ok.
        TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
            __FUNCTION__": device %p buffer is NULL\n",
            hubContext->WdfDevice);
    }
    if (Urb->UrbControlTransferEx.PipeHandle != NULL)
    {
        // request to interrupt endpoint.
        
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": device %p request to endpoint unsupported\n",
            hubContext->WdfDevice);
    }
    else
    {
        // request to control endpoint
        switch (packet.Packet.bRequest)
        {
        case USB_REQUEST_GET_DESCRIPTOR:
            transferRequest = "USB_REQUEST_GET_DESCRIPTOR";
            switch (packet.Packet.wValue.Bytes.HiByte)
            {
            case USB_DEVICE_DESCRIPTOR_TYPE:
                HubGetDeviceDescriptor(hubContext,
                    Request,
                    Urb,
                    buffer,
                    &packet);
                return;

            case USB_CONFIGURATION_DESCRIPTOR_TYPE:
                HubGetConfigurationDescriptor(hubContext,
                    Request,
                    Urb,
                    buffer,
                    &packet);
                return;

            case USB_STRING_DESCRIPTOR_TYPE:
                HubGetStringDescriptor(hubContext,
                    Request,
                    Urb,
                    buffer,
                    &packet);
                return;

            case USB_INTERFACE_DESCRIPTOR_TYPE:
                HubGetInterfaceDescriptor(hubContext,
                    Request,
                    Urb,
                    buffer,
                    &packet);
                return;

            case USB_ENDPOINT_DESCRIPTOR_TYPE:
                HubGetEndpointDescriptor(hubContext,
                    Request,
                    Urb,
                    buffer,
                    &packet);
                return;

            case USB_20_HUB_DESCRIPTOR_TYPE:
                HubGetHubDescriptor(hubContext,
                    Request,
                    Urb,
                    buffer,
                    &packet);
                return;

            default:
                break;
            }
            break;

        case USB_REQUEST_GET_CONFIGURATION:
            HubGetConfiguration(hubContext,
                    Request,
                    Urb,
                    buffer,
                    &packet);
            return;

        case USB_REQUEST_GET_STATUS:
            HubGetStatus(hubContext,
                    Request,
                    Urb,
                    buffer,
                    &packet);
            return;

        case USB_REQUEST_CLEAR_FEATURE:
            HubClearFeature(hubContext,
                    Request,
                    Urb,
                    &packet);
            return;

        case USB_REQUEST_SET_FEATURE:
            HubSetFeature(hubContext,
                    Request,
                    Urb,
                    &packet);
            return;

        case USB_REQUEST_SET_ADDRESS:
            transferRequest = "USB_REQUEST_SET_ADDRESS";
            break;
        case USB_REQUEST_SET_CONFIGURATION:
            transferRequest = "USB_REQUEST_SET_CONFIGURATION";
            break;
        default:
            break;
        }
    }
    
    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
        __FUNCTION__": device %p bRequest %s (%x) unsupported\n",
        hubContext->WdfDevice,
        transferRequest, 
        packet.Packet.bRequest);

    WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
    return;
}

/**
 * @brief process a URB_SELECT_CONFIGURATION request targeted at the HUB device. 
 *
 * @param[in] hubContext the context for this hub device.
 * @oaram[in] Request the Request object.
 * @param[in] Urb the URB provided by the Reauest.
 */
VOID
HubSelectConfiguration(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb)
{
    BOOLEAN noInterfaces = FALSE;
    
    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
                __FUNCTION__ "\n");

    if (Urb->UrbHeader.Length < sizeof(_URB_SELECT_CONFIGURATION))
    {
        if (Urb->UrbHeader.Length < NO_INTERFACE_LENGTH)
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
                __FUNCTION__ ": Device %p length %d less than minimum %d\n",
                hubContext->WdfDevice,
                Urb->UrbHeader.Length,
                NO_INTERFACE_LENGTH);

            WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
            return;
        }
        noInterfaces = TRUE;
    }

    PUSB_CONFIGURATION_DESCRIPTOR config = Urb->UrbSelectConfiguration.ConfigurationDescriptor;
    if (config == NULL)
    {
        //
        // this is ?most likely? NOT an actual usb config operation, but instead a
        // driver install operation.
        //
        TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
            __FUNCTION__ ": device %p NULL config ignored\n",
            hubContext->WdfDevice);
        WdfRequestComplete(Request, STATUS_SUCCESS);
        return;
    }
    
    //
    // validate the interface descriptor list
    //
    BOOLEAN setInterface = FALSE;
    USHORT InterfaceNumber = 0;
    USHORT Alternate = 0;
    if (noInterfaces == FALSE)
    {
        UCHAR interfacesInRequest = 0;
        PUSBD_INTERFACE_INFORMATION Interface = &Urb->UrbSelectConfiguration.Interface;
        PUCHAR start = (PUCHAR) Interface;
        PUCHAR end = ((PUCHAR) Urb) + Urb->UrbHeader.Length;
        while (start < end)
        {
            if (Interface->Length)
            {
                start += Interface->Length;
            }
            else
            {
                TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
                    __FUNCTION__ ": device %p interface %d length is zero\n",
                    hubContext->WdfDevice,
                    interfacesInRequest);
                WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
                return;
            }
            interfacesInRequest++;
            Interface = (PUSBD_INTERFACE_INFORMATION) start;
        }
        if (interfacesInRequest > config->bNumInterfaces)
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
                __FUNCTION__ ": device %p client wants %d interfaces from a config with %d\n",
                hubContext->WdfDevice,
                interfacesInRequest,
                config->bNumInterfaces);
            WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
            return;
        }
        
        if (interfacesInRequest > 1)
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
                __FUNCTION__ ": device %p hub only has one interface (%d)\n",
                hubContext->WdfDevice,
                interfacesInRequest);
            WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
            return;
        }
        setInterface = (interfacesInRequest == 1);
        Interface = &Urb->UrbSelectConfiguration.Interface;
        start = (PUCHAR) Interface;
        InterfaceNumber = Interface->InterfaceNumber;
        Alternate = Interface->AlternateSetting;
        PUSB_INTERFACE_DESCRIPTOR pInterfaceDescriptor =
            &hubContext->HubConfig.InterfaceDescriptor;

        Interface->InterfaceHandle = (USBD_INTERFACE_HANDLE) pInterfaceDescriptor;
        ULONG sizeNeeded = sizeof(USBD_INTERFACE_INFORMATION) - sizeof(USBD_PIPE_INFORMATION);
        if (pInterfaceDescriptor->bNumEndpoints)
        {
            sizeNeeded += (sizeof(USBD_PIPE_INFORMATION) * pInterfaceDescriptor->bNumEndpoints);
        }
        if (sizeNeeded > Interface->Length)
        {
            TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
                __FUNCTION__ ": device %p Invalid interface %d alternate %d request not enough space for %d pipes\n"
                " needed %d have %d\n",
                hubContext->WdfDevice,
                Interface->InterfaceNumber,
                Interface->AlternateSetting,
                pInterfaceDescriptor->bNumEndpoints,
                sizeNeeded,
                Interface->Length);

            WdfRequestComplete(Request, STATUS_UNSUCCESSFUL);
            return;
        }

        Interface->NumberOfPipes = pInterfaceDescriptor->bNumEndpoints;
        if (Interface->NumberOfPipes)
        {
            PIPE_DESCRIPTOR * pipeDesc = &hubContext->HubConfig.PipeDescriptor;

            PUSB_ENDPOINT_DESCRIPTOR endpoint = pipeDesc->endpoint;
            PUSBD_PIPE_INFORMATION pipe = &Interface->Pipes[0];

            pipe->MaximumPacketSize = endpoint->wMaxPacketSize;
            pipe->EndpointAddress = endpoint->bEndpointAddress;
            pipe->Interval = endpoint->bInterval;
            pipe->PipeType = (USBD_PIPE_TYPE) (endpoint->bmAttributes & 0x03);
            pipe->PipeHandle = (USBD_PIPE_HANDLE) pipeDesc;
        }

        Interface->Class = pInterfaceDescriptor->bInterfaceClass;
        Interface->SubClass = pInterfaceDescriptor->bInterfaceSubClass;
        Interface->Protocol = pInterfaceDescriptor->bInterfaceProtocol;
        Interface->Reserved = 0;
    }

    WdfRequestComplete(Request, STATUS_SUCCESS);
    return;
}

/**
* @brief process a URB_FUNCTION_GET_DESCRIPTOR_FROM_DEVICE request targeted at the HUB device.
*
* @param[in] hubContext the context for this hub device.
* @oaram[in] Request the Request object.
* @param[in] Urb the URB provided by the Reauest.
*/
VOID
HubProcessGetDescriptorUrb(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb)
{
    NTSTATUS Status = STATUS_UNSUCCESSFUL;
    ULONG bytesToCopy;

    PUCHAR buffer = (PUCHAR) Urb->UrbControlDescriptorRequest.TransferBuffer;
    if (Urb->UrbControlDescriptorRequest.TransferBufferMDL)
    {
        buffer = (PUCHAR) MmGetSystemAddressForMdlSafe(Urb->UrbControlDescriptorRequest.TransferBufferMDL,
            NormalPagePriority );
    }
    if (buffer == NULL)
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": device %p No buffer for descriptor\n",
            hubContext->WdfDevice);

        WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
        return;
    }

    switch (Urb->UrbControlDescriptorRequest.DescriptorType)
    {
    default:
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": device %p Invalid descriptor type %d\n",
            hubContext->WdfDevice,
            Urb->UrbControlDescriptorRequest.DescriptorType);
        break;

    case  USB_CONFIGURATION_DESCRIPTOR_TYPE:
        if (Urb->UrbControlDescriptorRequest.TransferBufferLength < sizeof(USB_CONFIGURATION_DESCRIPTOR))
        {
            Status = STATUS_INVALID_PARAMETER;
        }
        else
        {
            PUSB_CONFIGURATION_DESCRIPTOR config = 
                &hubContext->HubConfig.ConfigurationDescriptor;

            bytesToCopy = min(Urb->UrbControlDescriptorRequest.TransferBufferLength, config->wTotalLength);

            RtlCopyMemory(Urb->UrbControlDescriptorRequest.TransferBuffer,
                (PVOID) config,
                bytesToCopy);            

            Urb->UrbControlDescriptorRequest.TransferBufferLength = bytesToCopy;

            TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
                __FUNCTION__": Device %p config descriptor %d processed %d bytes returned\n",
                hubContext->WdfDevice,
                Urb->UrbControlDescriptorRequest.Index,
                bytesToCopy);
            Status = STATUS_SUCCESS;
        }
        break;

    case USB_DEVICE_DESCRIPTOR_TYPE:
        if (Urb->UrbControlDescriptorRequest.TransferBufferLength < sizeof(USB_DEVICE_DESCRIPTOR))
        {
            Status = STATUS_INVALID_PARAMETER;
        }
        else
        {
            bytesToCopy = min(Urb->UrbControlDescriptorRequest.TransferBufferLength, 
                hubContext->HubConfig.DeviceDescriptor.bLength);

            RtlCopyMemory(Urb->UrbControlDescriptorRequest.TransferBuffer,
                (PVOID) &hubContext->HubConfig.DeviceDescriptor,
                bytesToCopy);

            Urb->UrbControlDescriptorRequest.TransferBufferLength = bytesToCopy;
            TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
                __FUNCTION__": Device %p device descriptor processed %d bytes returned\n",
                hubContext->WdfDevice,
                bytesToCopy);
            Status = STATUS_SUCCESS;
        }
        break;

    case USB_INTERFACE_DESCRIPTOR_TYPE:
        if (Urb->UrbControlDescriptorRequest.TransferBufferLength < sizeof(USB_INTERFACE_DESCRIPTOR))
        {
            Status = STATUS_INVALID_PARAMETER;
        }
        else
        {
            bytesToCopy = min(Urb->UrbControlDescriptorRequest.TransferBufferLength, 
                hubContext->HubConfig.InterfaceDescriptor.bLength);

            RtlCopyMemory(Urb->UrbControlDescriptorRequest.TransferBuffer,
                (PVOID) &hubContext->HubConfig.InterfaceDescriptor,
                bytesToCopy);

            Urb->UrbControlDescriptorRequest.TransferBufferLength = bytesToCopy;
            TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
                __FUNCTION__": device %p interface descriptor processed %d bytes returned\n",
                hubContext->WdfDevice,
                bytesToCopy);
            Status = STATUS_SUCCESS;
        }
        break;

    case USB_ENDPOINT_DESCRIPTOR_TYPE:
        if (Urb->UrbControlDescriptorRequest.TransferBufferLength < sizeof(USB_ENDPOINT_DESCRIPTOR))
        {
            Status = STATUS_INVALID_PARAMETER;
        }
        else
        {
            bytesToCopy = min(Urb->UrbControlDescriptorRequest.TransferBufferLength, 
                hubContext->HubConfig.EndpointDescriptor.bLength);

            RtlCopyMemory(Urb->UrbControlDescriptorRequest.TransferBuffer,
                (PVOID) &hubContext->HubConfig.EndpointDescriptor,
                bytesToCopy);

            Urb->UrbControlDescriptorRequest.TransferBufferLength = bytesToCopy;
            TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
                __FUNCTION__": Device %p endpoint descriptor processed %d bytes returned\n",
                hubContext->WdfDevice,
                bytesToCopy);
            Status = STATUS_SUCCESS;
        }
        break;
    }
    WdfRequestComplete(Request, Status);
}

/**
 * @brief check if a pending HUB interrupt request should be completed with port
 * status change data.
 */
VOID
HubCheckStatusChange(
    IN PUSB_HUB_PDO_CONTEXT hubContext)
{
    if (hubContext->PortFeatureChange)
    {
        WDFREQUEST Request;
        NTSTATUS Status = WdfIoQueueRetrieveNextRequest(hubContext->StatusChangeQueue,
            &Request);

        if (NT_SUCCESS(Status))
        {
            PHUB_STATUS_CHANGE_CONTEXT context = RequestGetStatusChangeContext(Request);
            hubContext->ReportedStatus = hubContext->PortFeatureChange;
            context->Buffer[0] = 0x02; // port 1.

            TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_URB,
                __FUNCTION__": Device %p Request %p PortFeatureChange %x\n",
                hubContext->WdfDevice,
                Request,
                hubContext->PortFeatureChange);

            WdfRequestComplete(Request, STATUS_SUCCESS);
        }
        else switch (Status)
        {
        case STATUS_WDF_PAUSED:   // already unplugged.
        case STATUS_NO_MORE_ENTRIES:
            break;
        default:
            TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
                __FUNCTION__": Device %p WdfIoQueueRetrieveNextRequest error %x\n",
                hubContext->WdfDevice,
                Status);
        }
    }
}

/**
 * @brief park the hub status change interrupt request until
 * the status of the one port changes. 
 *
 * @param[in] hubContext the context for this hub device.
 * @param[in] Request the handle for the WDFREQUEST object containing the Urb.
 * @param[in] Urb the Interrupt URB targeted at the HUB device interrupt pipe.
 *
 */
VOID
HubProcessHubInterruptRequest(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    IN WDFREQUEST Request, 
    IN PURB Urb)
{
    PUCHAR buffer = (PUCHAR) Urb->UrbBulkOrInterruptTransfer.TransferBuffer;
    if (Urb->UrbBulkOrInterruptTransfer.TransferBufferMDL)
    {
        buffer = (PUCHAR) MmGetSystemAddressForMdlSafe(Urb->UrbBulkOrInterruptTransfer.TransferBufferMDL,
            NormalPagePriority );
    }

    if (buffer == NULL)
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": Device %p No buffer for descriptor\n",
            hubContext->WdfDevice);

        WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
        return;
    }

    if (Urb->UrbBulkOrInterruptTransfer.TransferBufferLength < 2)
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": Device %p TransferBufferLength %d too small\n",
            hubContext->WdfDevice,
            Urb->UrbBulkOrInterruptTransfer.TransferBufferLength);

        WdfRequestComplete(Request, STATUS_INVALID_PARAMETER);
        return;
    }
    //
    // Allocate a context for this request.
    //
    WDF_OBJECT_ATTRIBUTES  attributes;

    WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE(
        &attributes,
        HUB_STATUS_CHANGE_CONTEXT);

    PHUB_STATUS_CHANGE_CONTEXT context;

    NTSTATUS Status = WdfObjectAllocateContext(
        Request,
        &attributes,
        (PVOID*) &context);
    if (!NT_SUCCESS(Status))
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": Device %p request context allocation failed, %x\n",
            hubContext->WdfDevice,
            Status);

        WdfRequestComplete(Request, STATUS_INSUFFICIENT_RESOURCES);
        return;
    }

    context->Buffer = buffer;
    context->Length = Urb->UrbBulkOrInterruptTransfer.TransferBufferLength;

    RtlZeroMemory(buffer, Urb->UrbBulkOrInterruptTransfer.TransferBufferLength);

    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
            __FUNCTION__": Device %p Pipehandle %p Interrupt pipe %p length %d Queued\n",
            hubContext->WdfDevice,
            Urb->UrbBulkOrInterruptTransfer.PipeHandle,
            &hubContext->HubConfig.PipeDescriptor,
            context->Length);

    Status = WdfRequestForwardToIoQueue(Request, hubContext->StatusChangeQueue);
    
    if (!NT_SUCCESS(Status))
    {
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
            __FUNCTION__": Device %p WdfRequestForwardToIoQueue failed, %x\n",
            hubContext->WdfDevice,
            Status);

        WdfRequestComplete(Request, Status);
        return;
    }

    HubCheckStatusChange(hubContext);
}

/**
 * @brief the hub has an interrupt endpoint for status change
 * notification, so interrupt data URB's have to be handled here.
 * Also handled: URB_FUNCTION_CONTROL_TRANSFER_EX and 
 * URB_FUNCTION_GET_DESCRIPTOR_FROM_DEVICE.
 *
 * @note these are URBs targeted at the HUB itself not at the
 * ports owned by the HUB.
 *
 * @param[in] hubContext the context for this hub device.
 * @param[in] Request the handle for the WDFREQUEST object containing the Urb.
 * @param[in] Urb the URB targeted at the HUB device.
 *
 */
VOID
HubProcessUrb(
    IN PUSB_HUB_PDO_CONTEXT hubContext, 
    _In_ WDFREQUEST Request, 
    IN PURB Urb)
{
    NTSTATUS Status = STATUS_UNSUCCESSFUL;

    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
        __FUNCTION__": Device %p Request %p URB %p Function %x\n",
        hubContext->WdfDevice,
        Request,
        Urb,
        Urb->UrbHeader.Function);

    if (Urb->UrbHeader.Length < sizeof(_URB_HEADER))
    {
        WdfRequestComplete(Request, STATUS_UNSUCCESSFUL);

        TraceEvents(TRACE_LEVEL_ERROR, TRACE_URB,
            __FUNCTION__": Device %p Bad URB header\n",
            hubContext->WdfDevice);
        return;
    }    
    //
    // preset usb status to success!
    //
    Urb->UrbHeader.Status = USBD_STATUS_SUCCESS;
    PCHAR UrbFuncString = UrbFunctionToString(Urb->UrbHeader.Function);
    switch (Urb->UrbHeader.Function)
    {
    case URB_FUNCTION_CONTROL_TRANSFER_EX:
        HubControlTransferEx(hubContext,
            Request,
            Urb);
        return;

    case URB_FUNCTION_GET_DESCRIPTOR_FROM_DEVICE:
        HubProcessGetDescriptorUrb(hubContext, 
            Request, 
            Urb);        
        return;

    case URB_FUNCTION_SELECT_CONFIGURATION:
        HubSelectConfiguration(hubContext, 
            Request, 
            Urb);
        return;

    case URB_FUNCTION_BULK_OR_INTERRUPT_TRANSFER:
        HubProcessHubInterruptRequest(hubContext,
            Request,
            Urb);
        return;

    default:        
        TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
                __FUNCTION__": Device %p URB %s not implemented or not supported\n",
                hubContext->WdfDevice,
                UrbFuncString);
        Status = STATUS_UNSUCCESSFUL;
    }
    
    WdfRequestComplete(Request, Status);
}

/**
 * @brief process URB (Internal IOCTL) requests.
 * This function routes requests either to the parent 
 * controller device for processing or processes them locally.
 * For IOCTL_INTERNAL_USB_SUBMIT_URB the URB.UrbHeader.UsbdDeviceHandle
 * is used to decide if the HUB PDO or the Controller FDO should
 * process the request. All other IOCTL types are assumed to be
 * targeted at the HUB itself.
 *
 * @todo re-write this to be two routines, one for usbhub support,
 * the other one for non-usbhub support.
 *
 * @param[in] Queue the URB queue.
 * @param[in] Request the request handle.
 * @param[in] OutputBufferLength size of the output buffer.
 * @param[in] InputBufferLength size of the input buffer
 * @param[in] IoControlCode the function code.
 */
VOID
HubEvtIoInternalDeviceControl(
    _In_ WDFQUEUE Queue,
    _In_ WDFREQUEST Request,
    _In_ size_t OutputBufferLength,
    _In_ size_t InputBufferLength,
    _In_ ULONG IoControlCode
    )
{
    NTSTATUS Status = STATUS_NOT_IMPLEMENTED;
    UNREFERENCED_PARAMETER(OutputBufferLength);
    UNREFERENCED_PARAMETER(InputBufferLength);

    WDF_REQUEST_PARAMETERS parameters;
    WDF_REQUEST_PARAMETERS_INIT(&parameters);
    WdfRequestGetParameters(Request, &parameters);
    PUSB_HUB_PDO_CONTEXT hubContext = DeviceGetHubPdoContext(WdfIoQueueGetDevice(Queue));
    PUSB_FDO_CONTEXT fdoContext = GetFdoContextFromHubDevice(WdfIoQueueGetDevice(Queue));
    
    TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_QUEUE, 
        __FUNCTION__": Device %p Queue %p, Request %p IoControlCode %d\n", 
        hubContext->WdfDevice,
        Queue, 
        Request, 
        IoControlCode);

    BOOLEAN Unplugged = fdoContext->DeviceUnplugged || fdoContext->CtlrDisconnected || !fdoContext->PortConnected;

    switch (IoControlCode)
    {
    
    case IOCTL_INTERNAL_USB_GET_PARENT_HUB_INFO:
        {
            //
            // Itunes usbaapl.sys appears to think this is required.
            // 
            // Parameters.Others.Argument1 = PDEVICE_OBJECT * parent PDO
            // Parameters.Others.Argument2 = PULONG portNumber
            // Parameters.Others.Argument3 = IoControlCode
            // Parameters.Others.Argument4 = PDEVICE_OBJECT * rootHub PDO
            //
            PDEVICE_OBJECT * parent = (PDEVICE_OBJECT *) parameters.Parameters.Others.Arg1;
            PULONG portNumber = (PULONG) parameters.Parameters.Others.Arg2;
            PDEVICE_OBJECT * rootHubPdo  = (PDEVICE_OBJECT *) parameters.Parameters.Others.Arg4;
            if (parent)
            {
                *parent = WdfDeviceWdmGetDeviceObject(hubContext->WdfDevice); // HUB PDO
            }
            if (portNumber)
            {
                *portNumber = hubContext->Port;
            }
            if (rootHubPdo)
            {
                *rootHubPdo = WdfDeviceWdmGetDeviceObject(hubContext->WdfDevice); // Root HUB PDO 
            }

            TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
                __FUNCTION__": IOCTL_INTERNAL_USB_GET_PARENT_HUB_INFO for device %p p1 %p p2 %p p4 %p\n",
                hubContext->WdfDevice,
                parameters.Parameters.Others.Arg1,
                parameters.Parameters.Others.Arg2,
                parameters.Parameters.Others.Arg4);
            Status = STATUS_SUCCESS;
        }
        break;

    case IOCTL_INTERNAL_USB_GET_TOPOLOGY_ADDRESS:
        {
            PUSB_TOPOLOGY_ADDRESS topology = 
                (PUSB_TOPOLOGY_ADDRESS) parameters.Parameters.Others.Arg1;            

            TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
                __FUNCTION__": IOCTL_INTERNAL_USB_GET_TOPOLOGY_ADDRESS for device %p\n",
                hubContext->WdfDevice);

            if (topology)
            {
                topology->PciBusNumber = 0; // we could use the xen pci device info here
                topology->PciDeviceNumber = 0;
                topology->PciFunctionNumber = 0;
                topology->RootHubPortNumber = 0;
                RtlZeroMemory(topology->HubPortNumber, sizeof(topology->HubPortNumber));
            }
            Status = STATUS_SUCCESS;
        }
        break;

    case IOCTL_INTERNAL_USB_RESET_PORT:

        TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
                    __FUNCTION__": Device %p IOCTL_INTERNAL_USB_RESET_PORT\n",
                    hubContext->WdfDevice);

        if (!Unplugged)
        {
            // the hub needs to transition the port back to Enabled state when
            // the reset is complete. 
            KeClearEvent(&fdoContext->resetCompleteEvent);
            ForwardToParent(Request, hubContext);
            Request = NULL;

            if (KeGetCurrentIrql() != PASSIVE_LEVEL)
            {
                TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
                    __FUNCTION__": Device %p IOCTL_INTERNAL_USB_RESET_PORT at IRQL %d\n",
                    hubContext->WdfDevice,
                    KeGetCurrentIrql());
            }
            else
            {
                LARGE_INTEGER timeout;
                timeout.QuadPart = WDF_REL_TIMEOUT_IN_SEC(10);
                NTSTATUS waitStatus = KeWaitForSingleObject(&fdoContext->resetCompleteEvent,
                    Executive,
                    KernelMode,
                    FALSE,
                    &timeout);
                if (waitStatus == STATUS_SUCCESS)
                {
                    hubContext->PortFeatureStatus |= USB_PORT_STATUS_ENABLE;
                }
                else
                {
                    TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
                        __FUNCTION__": Device %p IOCTL_INTERNAL_USB_RESET_PORT error %x\n",
                        hubContext->WdfDevice,
                        waitStatus);
                }
            }
        }
        else
        {
            ForwardToParent(Request, hubContext);
            Request = NULL;
        }
        return;

        //
        // The following are only processed if providing a UsbHub PDO.
        //
    case IOCTL_INTERNAL_USB_GET_PORT_STATUS:
        {
            PULONG portStatus = (PULONG) parameters.Parameters.Others.Arg1;
            if (portStatus)
            {
                if (Unplugged)
                {
                    hubContext->PortFeatureStatus &= ~USB_PORT_STATUS_CONNECT;
                }
                *portStatus = (ULONG) hubContext->PortFeatureStatus;
                Status = STATUS_SUCCESS;
                TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
                    __FUNCTION__": device %p IOCTL_INTERNAL_USB_GET_PORT_STATUS port 1 status %x\n",
                    hubContext->WdfDevice,
                    *portStatus);
            }
            else
            {
                TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
                    __FUNCTION__": Device %p IOCTL_INTERNAL_USB_GET_PORT_STATUS invalid request\n",
                    hubContext->WdfDevice);
                Status = STATUS_INVALID_PARAMETER;
            }
        }
        break;

    case IOCTL_INTERNAL_USB_GET_DEVICE_HANDLE:
        //
        // Parameters.Others.Argument1 = "devicehandle" - but for what device?
        // the root hub?
        //
        if (parameters.Parameters.Others.Arg1)
        {
            PUSB_DEVICE_HANDLE * pHandle = 
                (PUSB_DEVICE_HANDLE *) parameters.Parameters.Others.Arg1;
            *pHandle = hubContext;

            TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
                __FUNCTION__": IOCTL_INTERNAL_USB_GET_DEVICE_HANDLE for device %p p1 %p p2 %p p4 %p\n",
                hubContext->WdfDevice,
                parameters.Parameters.Others.Arg1,
                parameters.Parameters.Others.Arg2,
                parameters.Parameters.Others.Arg4);
            Status = STATUS_SUCCESS;
        }
        break;

    case IOCTL_INTERNAL_USB_GET_DEVICE_HANDLE_EX:
        //
        // Parameters.Others.Argument1 = "devicehandle" - but for what device?
        // the root hub?
        // Parameters.Others.Argument2 = ???
        //
        if ((parameters.Parameters.Others.Arg1) &&
            (parameters.Parameters.Others.Arg2))
        {

            PUSB_DEVICE_HANDLE * pHandle = 
                (PUSB_DEVICE_HANDLE *) parameters.Parameters.Others.Arg1;
            *pHandle = hubContext;
            PVOID * arg2 = (PVOID *) parameters.Parameters.Others.Arg2;
            *arg2 = (PVOID) hubContext->Port; // ???

            TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
                __FUNCTION__": IOCTL_INTERNAL_USB_GET_DEVICE_HANDLE_EX for device %p p1 %p p2 %p p4 %p\n",
                hubContext->WdfDevice,
                parameters.Parameters.Others.Arg1,
                parameters.Parameters.Others.Arg2,
                parameters.Parameters.Others.Arg4);
            Status = STATUS_SUCCESS;
        }
        break;

    case IOCTL_INTERNAL_USB_GET_DEVICE_CONFIG_INFO:
        {
            PHUB_DEVICE_CONFIG_INFO configInfo = 
                (PHUB_DEVICE_CONFIG_INFO) parameters.Parameters.Others.Arg1;

            TraceEvents(TRACE_LEVEL_VERBOSE, TRACE_URB,
                __FUNCTION__": IOCTL_INTERNAL_USB_GET_DEVICE_CONFIG_INFO for device %p\n",
                hubContext->WdfDevice);
            configInfo->Version = 1;
            configInfo->Length = sizeof(HUB_DEVICE_CONFIG_INFO);
            configInfo->HubFlags.HubIsArmedWakeOnConnect = TRUE;
            configInfo->HubFlags.HubIsBusPowered = TRUE;
            configInfo->HubFlags.HubIsHighSpeed = TRUE;
            configInfo->HubFlags.HubIsHighSpeedCapable = TRUE;
            configInfo->HubFlags.HubIsMultiTt = FALSE;
            configInfo->HubFlags.HubIsMultiTtCapable = FALSE;
            configInfo->HubFlags.HubIsRoot = TRUE;

            configInfo->CompatibleIds.LanguageId = 0x0409;
            configInfo->CompatibleIds.Buffer = AllocAndQueryPropertyString(
                hubContext->WdfDevice,
                DevicePropertyCompatibleIDs,
                &configInfo->CompatibleIds.LengthInBytes);
            if (!configInfo->CompatibleIds.Buffer)
            {
                break;
            }

            configInfo->HardwareIds.LanguageId = 0x0409;
            configInfo->HardwareIds.Buffer = AllocAndQueryPropertyString(
                hubContext->WdfDevice,
                DevicePropertyHardwareID,
                &configInfo->HardwareIds.LengthInBytes);
            if (!configInfo->HardwareIds.Buffer)
            {
                ExFreePool(configInfo->CompatibleIds.Buffer);
                break;
            }

            configInfo->DeviceDescription.Buffer = NULL;
            Status = STATUS_SUCCESS;
        }
        break;

    case IOCTL_INTERNAL_USB_SUBMIT_URB:
        {
            PURB Urb = URB_FROM_REQUEST(Request);
            if (Urb->UrbHeader.UsbdDeviceHandle == 0)
            {
                //
                // this is targeted at the hub device.
                //
                HubProcessUrb(hubContext, Request, Urb);
                return;
            }
            else if (Urb->UrbHeader.UsbdDeviceHandle == (PVOID) -1)
            {
                //
                // does this ever happen? Is there a better test,
                // i.e. IsHandleValid(Urb->UrbHeader.UsbdDeviceHandle)
                //
                TraceEvents(TRACE_LEVEL_WARNING, TRACE_URB,
                    __FUNCTION__": Device %p Urb->UrbHeader.UsbdDeviceHandle %p invalid\n",
                    hubContext->WdfDevice,
                    Urb->UrbHeader.UsbdDeviceHandle);
                Status = STATUS_INVALID_PARAMETER;
                break;
            }
            else 
            {
                // 
                // this is targeted at the USB device.
                // @todo test handle for validity.
                // @todo (later) route based on handle.
                //
                ForwardToParent(Request, hubContext);
                return;
            }
        } 
        break;

    default:
        ForwardToParent(Request, hubContext);
        return;
    }

    WdfRequestComplete(Request, Status);
}

VOID
HubEvtIoStop(
    _In_ WDFQUEUE Queue,
    _In_ WDFREQUEST Request,
    _In_ ULONG ActionFlags
)
{
    TraceEvents(TRACE_LEVEL_VERBOSE, 
                TRACE_QUEUE, 
                __FUNCTION__": Queue 0x%p, Request 0x%p ActionFlags %d", 
                Queue, Request, ActionFlags);
    if (ActionFlags & WdfRequestStopRequestCancelable)
    {
        NTSTATUS Status = WdfRequestUnmarkCancelable(Request);
        if (Status == STATUS_CANCELLED)
        {
            return;
        }
    }
    WdfRequestComplete(Request, STATUS_CANCELLED);
    return;
}

VOID
HubStatusChangeEvtIoStop(
    _In_ WDFQUEUE Queue,
    _In_ WDFREQUEST Request,
    _In_ ULONG ActionFlags
)
{
    TraceEvents(TRACE_LEVEL_VERBOSE, 
                TRACE_QUEUE, 
                __FUNCTION__": Queue 0x%p, Request 0x%p ActionFlags %d", 
                Queue, Request, ActionFlags);
    if (ActionFlags & WdfRequestStopRequestCancelable)
    {
        NTSTATUS Status = WdfRequestUnmarkCancelable(Request);
        if (Status == STATUS_CANCELLED)
        {
            return;
        }
    }
    WdfRequestComplete(Request, STATUS_CANCELLED);
    return;
}

/**
 * @brief handles CreateFile operations for the usb controller.
 * This function basically exists only to log that a create occurred.
 *
 * @param[in] Device A handle to a framework device object. 
 * @param[in] Request A handle to a framework request object that represents a file creation request
 * @param[in] FileObject A handle to a framework file object that describes a file that is being opened for the specified request. 
 * This parameter is NULL if the driver has specified WdfFileObjectNotRequired for the FileObjectClass member of the WDF_FILEOBJECT_CONFIG structure. 
 */
VOID
HubEvtDeviceFileCreate (
    IN WDFDEVICE  Device,
    IN WDFREQUEST  Request,
    IN WDFFILEOBJECT  FileObject)
{
    PUSB_FDO_CONTEXT fdoContext = GetFdoContextFromHubDevice(Device);
    UNREFERENCED_PARAMETER(FileObject);

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_DEVICE,
        __FUNCTION__": %s Hub Device %p Controller Device %p\n",
        fdoContext->FrontEndPath,
        Device,
        fdoContext->WdfDevice);

    WdfRequestComplete(Request, STATUS_SUCCESS);
}

/**
 * @brief handles operations that must be performed when all of an application's 
 * accesses to a device have been closed.
 *
 * @param[in] FileObject The handle to the FileObject.
 */
VOID
HubEvtFileClose (
    IN WDFFILEOBJECT  FileObject)
{
    PUSB_FDO_CONTEXT fdoContext = GetFdoContextFromHubDevice(WdfFileObjectGetDevice(FileObject));

    TraceEvents(TRACE_LEVEL_INFORMATION, TRACE_DEVICE,
        __FUNCTION__": %s Hub Device %p Controller Device %p\n",
        fdoContext->FrontEndPath,
        WdfFileObjectGetDevice(FileObject),
        fdoContext->WdfDevice);
}

